/*
 * Copyright (c) 2014, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package com.sun.tools.sjavac.options;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;

import com.sun.tools.javac.main.CommandLine;
import com.sun.tools.sjavac.Transformer;

/**
 * This class is used to decode sjavac options.
 * See com.sun.tools.sjavac.options.Options for example usage.
 *
 *  <p><b>This is NOT part of any supported API.
 *  If you write code that depends on this, you do so at your own risk.
 *  This code and its internal interfaces are subject to change or
 *  deletion without notice.</b>
 */
public abstract class OptionHelper {

    /** Handle error */
    public abstract void reportError(String msg);

    /** Record a package exclusion pattern */
    public abstract void exclude(String excl);

    /** Record a package inclusion pattern */
    public abstract void include(String incl);

    /** Record a root of sources to be compiled */
    public abstract void sourceRoots(List<Path> path);

    /** Record a suffix + transformer */
    public abstract void addTransformer(String suffix, Transformer tr);

    /** Record a sourcepath to be used */
    public abstract void sourcepath(List<Path> path);

    /** Record a modulepath to be used */
    public abstract void modulepath(List<Path> path);

    /** Record a classpath to be used */
    public abstract void classpath(List<Path> path);

    /** Record the number of cores */
    public abstract void numCores(int parseInt);

    /** Record desired log level */
    public abstract void logLevel(String level);

    /** Record path for reference source list */
    public abstract void compareFoundSources(Path referenceList);

    /** Record a single permitted artifact */
    public abstract void permitArtifact(String f);

    /** Record the fact that unidentified artifacts are permitted */
    public abstract void permitUnidentifiedArtifacts();

    /** Record the fact that sources in the default package are permitted */
    public abstract void permitDefaultPackage();

    /** Record server configuration parameters */
    public abstract void serverConf(String serverConf);

    /** Record server launch configuration parameters */
    public abstract void startServerConf(String serverConf);

    /** Record some arguments to be passed on to javac */
    public abstract void javacArg(String... arg);

    /** Sets the destination directory for the compilation */
    public abstract void destDir(Path dir);

    /** Sets the directory for generated sources */
    public abstract void generatedSourcesDir(Path genSrcDir);

    /** Sets the directory for generated headers */
    public abstract void headerDir(Path dir);

    /** Sets the directory for state and log files generated by sjavac */
    public abstract void stateDir(Path dir);

    /** Sets the implicit policy */
    public abstract void implicit(String policy);


    /**
     * Traverses an array of arguments and performs the appropriate callbacks.
     *
     * @param args the arguments to traverse.
     */
    void traverse(String[] args) {
        Iterable<String> allArgs;
        try {
            allArgs = CommandLine.parse(List.of(args)); // Detect @file and load it as a command line.
        } catch (java.io.IOException e) {
            throw new IllegalArgumentException("Problem reading @"+e.getMessage());
        }
        ArgumentIterator argIter = new ArgumentIterator(allArgs);

        nextArg:
        while (argIter.hasNext()) {

            String arg = argIter.next();

            if (arg.startsWith("-")) {
                for (Option opt : Option.values()) {
                    if (opt.processCurrent(argIter, this))
                        continue nextArg;
                }

                javacArg(arg);

                // Does this javac argument take an argument? If so, don't
                // let it pass on to sjavac as a source root directory.
                for (com.sun.tools.javac.main.Option javacOpt : com.sun.tools.javac.main.Option.values()) {
                    if (javacOpt.matches(arg)) {
                        boolean takesArgument = javacOpt.hasArg();
                        boolean separateToken = !arg.contains(":") && !arg.contains("=");
                        if (takesArgument && separateToken)
                            javacArg(argIter.next());
                    }
                }
            } else {
                sourceRoots(Arrays.asList(Paths.get(arg)));
            }
        }
    }

    public static String unescapeCmdArg(String arg) {
        return arg.replaceAll("%20", " ")
                  .replaceAll("%2C", ",");
    }
}
