package com.mobilebasic;

/*
 * Mobile BASIC 1.9 Copyright (c) 2003-2010, David Firth
 *
 * This software is released as Open Source and you are free to modify it
 * and build derived versions for your own use. If you have made any changes
 * that you would like included in the main version then they should be
 * sent to the address below.
 *
 * Patches: E-mail to david@mobilebasic.com
 * Website: http://www.mobilebasic.com/
 */

import java.io.*;
import java.util.*;

/*
 * Space saving tips.
 *
 * 1. Make variables static if possible. Non static variables need 
 *    the this pointer to be loaded on the stack ... which takes an
 *    extra instruction.
 */

public class BASIC
{
    private static final boolean DEVELOPMENT_MODE = false;
    private static final boolean DEBUG = false;
    private static final int MAGIC = 0x4D420001;	// MB 0x0001
    
    public static final String versionString = "Mobile BASIC 1.9";

    private static final long MS_PER_DAY = (24 * 60 * 60 * 1000);
    private static Random random = new Random();
    private static Runtime runtime = Runtime.getRuntime();

    private static byte[] code = new byte[256];
    private static int codeLen = 0;

    private static byte[] sourceProg = null;
    private static int sourceSize = 0;
    private static int sourceLen = 0;

    private static boolean degFlag = false;
    private static boolean stepFlag = false;    // Used during runtime of BASIC program if the STEP operand is present in a FOR loop
    
    private static final int CONTROL_RETURN = 0;
    private static final int CONTROL_FORLOOP = 1;
    
    private static final byte CLASS_CONSTANT = 0;
    private static final byte CLASS_VARIABLE = 1;

    private static final byte TYPE_INTEGER = 0;
    private static final byte TYPE_FLOAT = 1;
    private static final byte TYPE_STRING = 2;

    private static int nvars;
    private static String[] varName = new String[256];
    private static byte[] varType = new byte[256];
    private static Object[] varObject = new Object[256];

    private static Enumeration dirEnum;

    static BasicSupport support;

    /*
     * FindLine returns the offset to the specified line.
     */
    
    public BASIC(BasicSupport support, int upperLimit)
    {
        BASIC.support = support;

        int currentSize = upperLimit;

        while (sourceProg == null)
        {
          try
          {
            sourceProg = new byte[currentSize];
            sourceSize = currentSize;
          }
          catch (OutOfMemoryError e)
          {
            currentSize -= 256;

            if (DEBUG)
              System.out.println ("sourceProg memory allocation failed, reducting requirement to: " + currentSize);

            if (currentSize <= 0)
              throw e;
          }
        }

        /*
         * This memory allocation method failed on the Siemens Again!
         * The problem this time is that as the program homes in on
         * the maximum value it can support the memory can get
         * fragmented. Thus when they converged it was possible
         * that the previous assignment to sourceProg could have
         * been zero.
         */
/*         
        int lowerLimit = 0;

        while ((upperLimit - lowerLimit) > 1)
        {
          int currentSize = (upperLimit + lowerLimit) / 2;

          // System.out.println("lowerLimit=" + lowerLimit + ", currentSize=" + currentSize + ", upperLimit=" + upperLimit);

          try
          {
            sourceProg = null;
            sourceProg = new byte[currentSize];
            sourceSize = currentSize;
            lowerLimit = currentSize;
          }
          catch (OutOfMemoryError e)
          {
            upperLimit = currentSize;
          }
        }
*/

        /*
         * Proguard was corrupting some of the unicode sequences inserted into tokenTable
         * Specifically, it corrupted \udd01 and converted in to 63
         * For the time being I've replaced this with \ucc01.
         *
         * The following is test code that is used to calculate a checksum on the unicode
         * values stored in tokenTable. If the checksums don't match after proguard has
         * been run then something is incorrect.
         */
        
        if (DEVELOPMENT_MODE)
        {
            int expected1;
            int expected2;
            int expected3;

            //expected1 = 0x40ded4;   // 0x3d66d0;
            //expected2 = 0x63b;
            //expected3 = 0x16944;

            expected1 = 0x440ad7;
            expected2 = 0x64c;
            expected3 = 0x169a4;
        
            int sum1 = 0;
            int sum2 = 0;
            int sum3 = 0;
        
            for (int i=0;i<tokenTable.length;i++)
            {
                int word1 = tokenTable[i].charAt(0) & 0xffff;
                int word2 = tokenTable[i].charAt(1) & 0xffff;
                int word3 = tokenTable[i].charAt(2) & 0xffff;
            
                sum1 += word1;
                sum2 += word2;
                sum3 += word3;
            }
        
            System.out.println("tokenTable Checksums: 0x" +
                               Integer.toHexString(sum1) + " 0x" +
                               Integer.toHexString(sum2) + " 0x" +
                               Integer.toHexString(sum3));
        
            System.out.println("Differences: 0x" +
                               Integer.toHexString(expected1-sum1) + " 0x" +
                               Integer.toHexString(expected2-sum2) + " 0x" +
                               Integer.toHexString(expected3-sum3));
        }
    }

  /*
   * GetToken() skips past any white space and returns the
   * next token or null if a valid token cannot be found.
   *
   * Extract Constant Tokens
   *   "ddddhhdhdhd"	STRING CONSTANT
   *   123		INTEGER CONSTANT
   *   123.0E+2		FLOAT CONSTANT
   *
   * Extract Token Terminated By:-
   *   Following a '%' or '$' character
   *   The next non-alphanumeric character
   *
   * JDJDJDDJ$		STRING IDENTIFIER
   * JDJDJDJD%		INTEGER IDENTIFIER
   * FJFJFJFF		FLOAT IDENTIFIER
   * IF * / etc		RESEVERED TOKEN
   *
   * What can terminate a token
   *   A space character
   *     A = 4
   *     IF A=4 THEN xxxx
   *   A non alphanumeric character
   *     A=4
   *     IFA=4 
   */

  private static String line;
  private static int lineLen;
  private static int lineOffset;
  private static String nextToken;

  private static void PutToken(String token)
  {
    nextToken = token;
    //lineOffset -= nextToken.length();
  }

  private static String GetToken()
  {
    String token = null;

    if (nextToken != null)
    {
      //lineOffset += nextToken.length();
      token = nextToken;
      nextToken = null;
    }
    else
    {
      while (lineOffset < lineLen)
      {
        if (line.charAt(lineOffset) != ' ')
          break;
        lineOffset++;
      }

      if (lineOffset < lineLen)
      {
        StringBuffer sb = new StringBuffer();

        boolean forceUpperCase = false;
        boolean tokenValid = false;

        char ch = line.charAt(lineOffset++);
        if (ch == '\"')				// Must be a String Constant
        {
          boolean EscFlag = false;
          
          sb.append(ch);

          while (lineOffset < lineLen)
          {
            ch = line.charAt(lineOffset++);

            if (!EscFlag)
            {
              if (ch == '\"')
              {
                sb.append(ch);
                tokenValid = true;
                break;
              }
              else if (ch == '\\')
              {
                EscFlag = true;
                continue;
              }
            }

            sb.append(ch); 
            EscFlag = false;
          }
        }
        else if ((ch >= '0') && (ch <= '9'))	// Must be a Numeric Constant
        {
          tokenValid = true;			// Just in case we reach EOL

          int state = 0;

          sb.append(ch);

        whileLoop:
          while(lineOffset < lineLen)
          {
            ch = line.charAt(lineOffset);

            switch (state)
            {
              case -1 :
                break whileLoop;
  
              case 0 :				// Before Decimal Point
                if ((ch >= '0') && (ch <= '9'))
                {
                  sb.append(ch);
                  lineOffset++;
                }
                else if (ch == '.')			// Decimal point?
                {
                  sb.append(ch);
                  lineOffset++;
                  state = 1;
                }
                else if ((ch == 'E') ||		// E or e?
                         (ch == 'e'))
                {
                  tokenValid = false;
                  sb.append('E');
                  lineOffset++;
                  state = 2;
                }
                else
                {
                  state = -1;			// 123
                }
                break;
              case 1 :				// After Decimal Point
                if ((ch >= '0') && (ch <= '9'))
                {
                  sb.append(ch);
                  lineOffset++;
                }
                else if ((ch == 'E') ||		// E or e?
                         (ch == 'e'))
                {
                  tokenValid = false;
                  sb.append('E');
                  lineOffset++;
                  state = 2;
                }
                else
                {
                  state = -1;			// 123.[nnn]
                }
                break;
              case 2 :
                if ((ch =='+') || (ch =='-'))
                {
                  sb.append(ch);
                  lineOffset++;
                  state = 3;
                }
                else
                {
                  tokenValid = false;		// 123[.[nnn]]E (missing +|-)
                  state = -1;
                }
                break;
              case 3 :
                if ((ch >= '0') && (ch <= '9'))
                {
                  tokenValid = true;
                  sb.append(ch);
                  lineOffset++;
                  state = 4;
                }
                else
                {
                  tokenValid = false;		// 123[.[nnn]]E+ (missing val)
                  state = -1;
                }
                break;
              case 4 :
                if ((ch >= '0') && (ch <= '9'))
                {
                  sb.append(ch);
                  lineOffset++;
                }
                else
                {
                  state = -1;			// OK
                }
                break;
            }
          }
        }
        else if (((ch >= 'a') && (ch <= 'z')) ||	// Keyword | Identifier
                 ((ch >= 'A') && (ch <= 'Z')))
        {
          sb.append(ch);
          forceUpperCase = true;
          tokenValid = true;

          while (lineOffset < lineLen)
          {
            ch = line.charAt(lineOffset);
            if (((ch >= 'a') && (ch <= 'z')) ||
                ((ch >= 'A') && (ch <= 'Z')) ||
                ((ch >= '0') && (ch <= '9')))
            {
              sb.append(ch);
              lineOffset++;
            }
            else if ((ch == '$') || (ch == '%'))
            {
              sb.append(ch);
              lineOffset++;
              break;
            }
            else
            {
              break;
            }
          }
        }
        else					// Symbol
        {
          /*
           * + - * / & | ^ = ( ) ,
           * <> < <=
           * >= >
           */

          //if ("+-*/&|^=(),#:".indexOf(ch) != -1)
          if ("+-*/^=(),#:".indexOf(ch) != -1)
          {
            sb.append(ch);
            tokenValid = true;
          }
          else if (ch == '<')
          {
            sb.append(ch);
            if (lineOffset < lineLen)
            {
              ch = line.charAt(lineOffset);
              if ((ch == '=') || (ch == '>'))
              {
                sb.append(ch);
                lineOffset++;
              }
            }
            tokenValid = true;
          }
          else if (ch == '>')
          {
            sb.append(ch);
            if (lineOffset < lineLen)
            {
              ch = line.charAt(lineOffset);
              if (ch == '=')
              {
                sb.append(ch);
                lineOffset++;
              }
            }
            tokenValid = true;
          }
        }

        if (tokenValid)
        {
          token = sb.toString();
          if ((token != null) && forceUpperCase)
              token = token.toUpperCase();
        }
      }
    }

    return token;
  }

    public static Vector ReadLines(DataInput dataInput)
    {
        Vector vector = null;
        
        if (dataInput != null)
        {
            String s;
            
            while ((s = ReadLine(dataInput)) != null)
            {
                if (vector == null)
                    vector = new Vector();
                
                vector.addElement(s);
            }
        }
        
        return vector;
    }
    
    public static String ReadLine(DataInput dataInput)
    {
        String s = null;
        
        if (dataInput != null)
        {
            StringBuffer sb = new StringBuffer();
            int ch = -1;

            try
            {
                while ((ch = dataInput.readByte()) != -1)
                {
                    if (ch == '\n')
                        break;
                    else if (ch != '\r')
                        sb.append((char)ch);
                }
            }
            catch (EOFException e)
            {
            }
            catch (IOException e)
            {
                e.printStackTrace();
            }

            /*
             * Get string to return if length() > 0. We must also hanlde
             * the case where a '\n' was on a line by itself - in this
             * case length() would be zero yet we are not at the end of
             * file. Set return string to ""
             */
                
            if (sb.length() > 0)
            {
                s = sb.toString();
            }
            else
            {
                if (ch == '\n')
                    s = "";
            }
        }
        
        //System.out.println("ReadLine: " + s);

        return s;
    }
    
    public static boolean Enter(DataInput dataInput)
    {
        boolean errorFlag = false;

        if (dataInput != null)
        {
            String s;
            
            while ((s = ReadLine(dataInput)) != null)
            {
                //System.out.println("s=" + s);
                boolean okFlag = parseLine(s, false);
                if (!okFlag)
                {
                  errorFlag = true;
                  break;
                }
            }
        }

        return errorFlag;
     }

     public static void LoadFrom(DataInput dataInput)
     {
       try
       {
         if (dataInput.readInt() != MAGIC)
           throw new BasicError(BasicError.INCOMPATIBLE_FILE_FORMAT, "Incompatible file format");

         nvars = dataInput.readShort();

         for (int i=0;i<nvars;i++)
         {
           varName[i] = dataInput.readUTF();
           varType[i] = dataInput.readByte();
           switch (varType[i])
           {
             case TYPE_STRING :
               varObject[i] = new String[1];
               break;
             case TYPE_INTEGER :
             case TYPE_FLOAT :
               varObject[i] = new int[1];
               break;
             default :
               throw new BasicError(BasicError.INTERNAL_ERROR, "");
           }
         }

         sourceLen = dataInput.readShort();
         if (sourceLen <= sourceSize)
           dataInput.readFully(sourceProg, 0, sourceLen);
         else
           throw new BasicError(BasicError.OUT_OF_MEMORY, "Out of memory");
       }
       catch (IOException e)
       {
         throw new BasicError(BasicError.IO_ERROR, "");
       }
     }

     public static boolean SaveTo(DataOutput dataOutput)
     {
       boolean errorFlag;

       try
       {
         dataOutput.writeInt(MAGIC);

         dataOutput.writeShort(nvars);

         for (int i=0;i<nvars;i++)
         {
           dataOutput.writeUTF(varName[i]);
           dataOutput.writeByte(varType[i]);
         }

         dataOutput.writeShort(sourceLen);
         dataOutput.write(sourceProg, 0, sourceLen);

         errorFlag = false;
       }
       catch (Throwable t)
       {
         errorFlag = true;
       }

       return errorFlag;
     }

    private static int FindLine(int lno, byte[] searchProg, int searchLen)
    {
        //System.out.println("FindLine: Searching for line " + lno);
        
        int linePC = 0;

        while (linePC < searchLen)
        {
            int lineNum;
            int lineLen;

            lineNum = ((searchProg[linePC] & 0xff) << 8) +
                       (searchProg[linePC+1] & 0xff);
            lineLen = (searchProg[linePC+2] & 0xff);

            if (lno == lineNum)
            {
                //System.out.println("FindLine: Found line " + lno);
                //lastLine[lno] = linePC;
                return linePC;
            }
            else if (lineNum > lno)
            {
                return -1;
            }
 
            linePC += lineLen;
        }

        //System.out.println("FindLine: Line " + lno + " not found");

        return -1;
    }

    /*
     * InsertLine insert the specified line into the program.
     */

    private static void InsertLine(int lno, int len, byte[] buf)
    {
        RemoveLine(lno);

        int insertPos = 0;

        while (insertPos < sourceLen)
        {
            int lineNum;
            int lineLen;

            lineNum = ((sourceProg[insertPos] & 0xff) << 8) +
                       (sourceProg[insertPos+1] & 0xff);
            lineLen = (sourceProg[insertPos+2] & 0xff);

            if (lineNum > lno)
                break;
 
            insertPos += lineLen;
        }

        if ((sourceLen + len) <= sourceSize)
        {
          if (insertPos < sourceLen)
          {
              int src = sourceLen;
              sourceLen += len;
              int dest = sourceLen;

              while (src > insertPos)
                  sourceProg[--dest] = sourceProg[--src];
          }
          else
          {
              sourceLen += len;
          }

          buf[0] = (byte)((lno >> 8) & 0xff);
          buf[1] = (byte)(lno & 0xff);
          buf[2] = (byte)(len);
        
          for (int i=0;i<len;i++)
            sourceProg[insertPos++] = buf[i];
        }
        else
        {
           throw new BasicError(BasicError.OUT_OF_MEMORY, "Out of memory");
        }
    }

    /*
     * RemoveLine removes the specified line from the program
     */

    private static void RemoveLine(int lno)
    {
        int linePC = FindLine(lno, sourceProg, sourceLen);
        if (linePC != -1)
        {
            int dest = linePC;
            int len = (sourceProg[linePC + 2] & 0xff);
            int src = linePC + len;

            while (src < sourceLen)
                sourceProg[dest++] = sourceProg[src++];

            sourceLen = dest;
        }
    }

    public static void New()
    {
      sourceLen = 0;

      nvars = 0;
      for (int i=0;i<varName.length;i++)
      {
        varName[i] = null;
        //varType[i] = 0;
        varObject[i] = null;
      }
    }
    
    private static final byte tokSTOP = 0;
    private static final byte tokPOP = 1;
    private static final byte tokRETURN = 2;
    private static final byte tokEND = 3;
    private static final byte tokNEW = 4;
    private static final byte tokRUN = 5;
    private static final byte tokDIR = 6;
    private static final byte tokDEG = 7;
    private static final byte tokRAD = 8;
    private static final byte tokBYE =  9;
    private static final byte tokGOTO = 10;
    private static final byte tokGOSUB = 11;
    private static final byte tokSLEEP = 12;
    private static final byte tokPRINT = 13;
    private static final byte tokREM = 14;
    private static final byte tokDIM = 15;
    private static final byte tokIF = 16;
    private static final byte tokTHEN = 17;
    private static final byte tokCLS = 18;
    private static final byte tokPLOT = 19;
    private static final byte tokDRAWLINE = 20;
    private static final byte tokFILLRECT = 21;
    private static final byte tokDRAWRECT = 22;
    private static final byte tokFILLROUNDRECT = 23;
    private static final byte tokDRAWROUNDRECT = 24;
    private static final byte tokFILLARC = 25;
    private static final byte tokDRAWARC = 26;
    private static final byte tokDRAWSTRING = 27;
    private static final byte tokSETCOLOR = 28;
    private static final byte tokBLIT = 29;
    private static final byte tokFOR = 30;
    private static final byte tokTO = 31;
    private static final byte tokSTEP = 32;
    private static final byte tokNEXT = 33;
    private static final byte tokINPUT = 34;
    private static final byte tokLIST = 35;
    private static final byte tokENTER = 36;
    private static final byte tokLOAD = 37;
    private static final byte tokSAVE = 38;
    private static final byte tokDELETE = 39;
    private static final byte tokEDIT = 40;
    private static final byte tokTRAP = 41;
    private static final byte tokOPEN = 42;
    private static final byte tokCLOSE = 43;
    private static final byte tokNOTE = 44;
    private static final byte tokPOINT = 45;
    private static final byte tokPUT = 46;
    private static final byte tokGET = 47;
    private static final byte tokDATA = 48;
    private static final byte tokRESTORE = 49;
    private static final byte tokREAD = 50;
    private static final byte tokEQ = 51;
    private static final byte tokNE = 52;
    private static final byte tokLT = 53;
    private static final byte tokLE = 54;
    private static final byte tokGT = 55;
    private static final byte tokGE = 56;
    private static final byte tokLBRACKET = 57;
    private static final byte tokRBRACKET = 58;
    private static final byte tokCOMMA = 59;
    private static final byte tokADD = 60;
    private static final byte tokSUB = 61;
    private static final byte tokUMINUS = 62;
    private static final byte tokMULT = 63;
    private static final byte tokDIV = 64;
    private static final byte tokPOWER = 65;
    private static final byte tokBITAND = 66;
    private static final byte tokBITOR = 67;
    private static final byte tokBITXOR = 68;
    private static final byte tokLOGNOT = 69;
    private static final byte tokLOGAND = 70;
    private static final byte tokLOGOR = 71;
    private static final byte tokSCREENWIDTH = 72;
    private static final byte tokSCREENHEIGHT = 73;
    private static final byte tokISCOLOR = 74;
    private static final byte tokNUMCOLORS = 75;
    private static final byte tokSTRINGWIDTH = 76;
    private static final byte tokSTRINGHEIGHT = 77;
    private static final byte tokLEFT$ = 78;
    private static final byte tokMID$ = 79;
    private static final byte tokRIGHT$ = 80;
    private static final byte tokCHR$ = 81;
    private static final byte tokSTR$ = 82;
    private static final byte tokLEN = 83;
    private static final byte tokASC= 84;
    private static final byte tokVAL= 85;
    private static final byte tokUP = 86;
    private static final byte tokDOWN = 87;
    private static final byte tokLEFT = 88;
    private static final byte tokRIGHT = 89;
    private static final byte tokFIRE = 90;
    private static final byte tokGAMEA = 91;
    private static final byte tokGAMEB = 92;
    private static final byte tokGAMEC = 93;
    private static final byte tokGAMED = 94;
    private static final byte tokDAYS = 95;
    private static final byte tokMILLISECONDS = 96;
    private static final byte tokYEAR = 97;
    private static final byte tokMONTH = 98;
    private static final byte tokDAY = 99;
    private static final byte tokHOUR = 100;
    private static final byte tokMINUTE = 101;
    private static final byte tokSECOND = 102;
    private static final byte tokMILLISECOND = 103;
    private static final byte tokRND = 104;
    private static final byte tokERR = 105;
    private static final byte tokFRE = 106;
    private static final byte tokMOD = 107;
    private static final byte tokEDITFORM = 108;
    private static final byte tokGAUGEFORM = 109;
    private static final byte tokCHOICEFORM = 110;
    private static final byte tokDATEFORM = 111;
    private static final byte tokMESSAGEFORM = 112;
    private static final byte tokLOG = 113;
    private static final byte tokEXP = 114;
    private static final byte tokSQR = 115;
    private static final byte tokSIN = 116;
    private static final byte tokCOS = 117;
    private static final byte tokTAN = 118;
    private static final byte tokASIN = 119;
    private static final byte tokACOS = 120;
    private static final byte tokATAN = 121;
    private static final byte tokABS = 122;
    
    private static final byte tokFOREQ= 123;
    private static final byte tokHASH = 124;
    private static final byte tokPRINTHASH = 125;
    private static final byte tokINPUTHASH = 126;

    private static final int tokCOLON = 127;

    private static final int tokGELGRAB = 128;
    private static final int tokDRAWGEL = 129;
    private static final int tokSPRITEGEL = 130;
    private static final int tokSPRITEMOVE = 131;
    private static final int tokSPRITEHIT = 132;

    private static final int tokREADDIR$ = 133;
    private static final int tokPROPERTY$ = 134;
    private static final int tokGELLOAD = 135;
    private static final int tokGELWIDTH = 136;
    private static final int tokGELHEIGHT = 137;

    /*
     * Special Tokens
     */
    
    private static final int tokASSIGN = 246;
    private static final int tokMAKEREF = 247;
    private static final int tokBYTE = 248;	// -128 -> 127
    private static final int tokUBYTE = 249;	// 0 -> 255
    private static final int tokUWORD = 250;
    private static final int tokINTEGER = 251;
    private static final int tokVARIABLE = 252;
    private static final int tokSTRING = 253;
    private static final int tokFLOAT = 254;
    private static final int tokEOS = 255;

    /*
     * MAKEREF was introduced because there was a problem identifing references on the stack
     * The original method was to assume a constant on the top of stack was an index to the variable
     * that follows.
     *
     * e.g. A(5) -> A 5 (5 is the reference to A)
     *
     * If 5 is missing then A was assumed to be a variable without an Index.
     *
     * Unfortunately A(N) -> A N
     *
     * Since there is no number on the stack N was interpreted to be a variable without an Index.
     *
     * When accessing arrays parseRValue can either insert an array reference or leave the variable
     * and index on the stack. From now onwards parseR
     */
  /*
   * reservedTokenList consists of several fields:-
   *
   * Word 1
   *   Nibble 3
   *     Go on stack priority
   *   Nibble 2
   *     Come off stack priority
   *   Nibble 1
   *     Bit 7 - Print before this token while generating a listing
   *     Bit 6 - Print space after this token while generating a listing
   *     Bit 5
   *     Bit 4
   *   Nibble 0
   *     Token Group - lookupToken only matches items in specified group
   *       Group 0 - Regulary Keywords
   *       Group 1 - Expression
   *       Group 2 - Items following keyword. e.g. THEN, TO and STEP (Tend to have priorities == Comma)
   *
   * Word 2 - Token Type (Bit Vector representing 16 Token Types - used for Syntax Checking)
   *   Bit 15       Unary Operator      0x8000
   *   Bit 14       Binary Operator     0x4000
   *   Bit 13       Left Bracket        0x2000
   *   Bit 12       Right Bracket       0x1000
   *   Bit 11       Comma               0x0800
   *   Bit 10       Function            0x0400
   *   Bit  9       Variable            0x0200
   *   Bit  8       Constant            0x0100
   *   Bit  7       EOS                 0x0080
   *   Bit  6
   *   Bit  5
   *   Bit  4       #                   0x0010
   *   Bit  3       STEP                0x0008
   *   Bit  2       TO                  0x0004
   *   Bit  1       THEN                0x0002
   *   Bit  0       Keyword             0x0001
   *
   * Word 3 - Valid Next Tokens Types
   *
   * PUSH STATE
   * POP STATE      (Would need some form of indicator to say we have returned to state)
   *
   * Current State  Next States
   * START          UNARY | LBRACKET | FUNCTION | VARIABLE | CONSTANT   %1010 0111 0000 0000    0xa700
   * UNARY          LBRACKET | FUNCTION | VARIABLE | CONSTANT           %0010 0111 0000 0000    0x2700
   * BINARY         UNARY | LBRACKET | FUNCTION | VARIABLE | CONSTANT   %1010 0111 0000 0000    0xa700
   * LBRACKET       UNARY | LBRACKET | FUNCTION | VARIABLE | CONSTANT   %1010 0111 0000 0000    0xa700
   * RBRACKET       BINARY | RBRACKET | COMMA | EOS                     %0101 1000 1000 0000    0x5880
   * COMMA          UNARY | LBRACKET | FUNCTION | VARIABLE | CONSTANT   %1010 0111 0000 0000    0xa700
   * FUNCTION       LBRACKET                                            %0010 0000 0000 0000    0x2000
   * VARIABLE       BINARY | LBRACKET | RBRACKET | COMMA | EOS          %0111 1000 1000 0000    0x7880
   * CONSTANT       BINARY | RBRACKET | COMMA | EOS                     %0101 1000 1000 0000    0x5880
   */
  
  private static final String[] tokenTable =
  {
    "\u1100\u0001\u0080STOP",
    "\u1100\u0001\u0080POP",
    "\u1100\u0001\u0080RETURN",
    "\u1100\u0001\u0080END",
    "\u1100\u0001\u0080NEW",
    "\u1100\u0001\u0080RUN",
    "\u1100\u0001\u0080DIR",
    "\u1100\u0001\u0080DEG",
    "\u1100\u0001\u0080RAD",
    "\u1100\u0001\u0080BYE",
    "\u1140\u0001\u0000GOTO",
    "\u1140\u0001\u0000GOSUB",
    "\u1140\u0001\u0000SLEEP",
    "\u0040\u0001\ua700PRINT",
    "\u1100\u0001\u0000REM",            // We shouldn't put a space after REM since list does it for us. REM comment[space would be here]
    "\u1140\u0001\u0200DIM",
    "\u1140\u0001\u0000IF",
    "\u11c2\u0001\u0000THEN",
    "\u1100\u0001\u0000CLS",
    "\u1140\u0001\u0000PLOT",
    "\u1140\u0001\u0000DRAWLINE",
    "\u1140\u0001\u0000FILLRECT",       // 20
    "\u1140\u0001\u0000DRAWRECT",
    "\u1140\u0001\u0000FILLROUNDRECT",
    "\u1140\u0001\u0000DRAWROUNDRECT",
    "\u1140\u0001\u0000FILLARC",
    "\u1140\u0001\u0000DRAWARC",
    "\u1140\u0001\u0000DRAWSTRING",
    "\u1140\u0001\u0000SETCOLOR",
    "\u1140\u0001\u0000BLIT",
    "\uf240\u0001\u0000FOR",    // FOR Same Priority as (
    "\u43c2\u0001\u0000TO",     // TO Same Priority as ,
    "\u43c2\u0001\u0000STEP",   // STEP Same Priority as ,
    "\u1140\u0001\u0000NEXT",
    "\u1140\u0001\u0000INPUT",
    "\u1140\u0001\u0080LIST",
    "\u1140\u0001\u0000ENTER",
    "\u1140\u0001\u0000LOAD",
    "\u1140\u0001\u0000SAVE",
    "\u1140\u0001\u0000DELETE",
    "\u1140\u0001\ua700EDIT",
    "\u1140\u0001\u0000TRAP",       // 40
    "\u1140\u0001\u0000OPEN",
    "\u1140\u0001\u0000CLOSE",
    "\u1140\u0001\u0000NOTE",
    "\u1140\u0001\u0000POINT",
    "\u1140\u0001\u0000PUT",
    "\u1140\u0001\u0000GET",
    "\u1100\u0001\u0000DATA",       // We shouldn't put a space after DATA since list does it for us. DATA item[space would be here]
    "\u1140\u0001\u0000RESTORE",
    "\u1140\u0001\u0000READ",

    "\u8801\u0040\u00a7=",
    "\u8801\u0040\u00a7<>",
    "\u8801\u0040\u00a7<",
    "\u8801\u0040\u00a7<=",
    "\u8801\u0040\u00a7>",
    "\u8801\u0040\u00a7>=",
    "\uf201\u0020\u00a7(",
    "\u2e01\u0010\u0158)",          // \u4e01
    "\u4301\u0008\u00a7,",
    "\u9901\u0040\u00a7+",
    "\u9901\u0040\u00a7-",          // 60
    // "\udd01\u0080\u0027-",    // (Unary Minus)
    "\ucc01\u0080\u0027-",    // (Unary Minus)
    "\uaa01\u0040\u00a7*",
    "\uaa01\u0040\u00a7/",
    "\ubb01\u0040\u00a7^",      // Raise to power
    "\u88c1\u0040\u00a7BITAND",
    "\u88c1\u0040\u00a7BITOR",
    "\u88c1\u0040\u00a7BITXOR",
    "\u77c1\u0080\u0027NOT",
    "\u66c1\u0040\u00a7AND",
    "\u55c1\u0040\u00a7OR",
    "\uf201\u0004\u0020SCREENWIDTH",
    "\uf201\u0004\u0020SCREENHEIGHT",
    "\uf201\u0004\u0020ISCOLOR",
    "\uf201\u0004\u0020NUMCOLORS",
    "\uf201\u0004\u0020STRINGWIDTH",
    "\uf201\u0004\u0020STRINGHEIGHT",
    "\uf201\u0004\u0020LEFT$",
    "\uf201\u0004\u0020MID$",
    "\uf201\u0004\u0020RIGHT$",
    "\uf201\u0004\u0020CHR$",       // 80
    "\uf201\u0004\u0020STR$",
    "\uf201\u0004\u0020LEN",
    "\uf201\u0004\u0020ASC",
    "\uf201\u0004\u0020VAL",
    "\uf201\u0004\u0020UP",
    "\uf201\u0004\u0020DOWN",
    "\uf201\u0004\u0020LEFT",
    "\uf201\u0004\u0020RIGHT",
    "\uf201\u0004\u0020FIRE",
    "\uf201\u0004\u0020GAMEA",
    "\uf201\u0004\u0020GAMEB",
    "\uf201\u0004\u0020GAMEC",
    "\uf201\u0004\u0020GAMED",
    "\uf201\u0004\u0020DAYS",
    "\uf201\u0004\u0020MILLISECONDS",
    "\uf201\u0004\u0020YEAR",
    "\uf201\u0004\u0020MONTH",
    "\uf201\u0004\u0020DAY",
    "\uf201\u0004\u0020HOUR",
    "\uf201\u0004\u0020MINUTE",         // 100
    "\uf201\u0004\u0020SECOND",
    "\uf201\u0004\u0020MILLISECOND",
    "\uf201\u0004\u0020RND",
    "\uf201\u0004\u0020ERR",
    "\uf201\u0004\u0020FRE",
    "\uf201\u0004\u0020MOD",
    "\uf201\u0004\u0020EDITFORM",
    "\uf201\u0004\u0020GAUGEFORM",
    "\uf201\u0004\u0020CHOICEFORM",
    "\uf201\u0004\u0020DATEFORM",
    "\uf201\u0004\u0020MESSAGEFORM",
    "\uf201\u0004\u0020LOG",
    "\uf201\u0004\u0020EXP",
    "\uf201\u0004\u0020SQR",
    "\uf201\u0004\u0020SIN",
    "\uf201\u0004\u0020COS",
    "\uf201\u0004\u0020TAN",
    "\uf201\u0004\u0020ASIN",
    "\uf201\u0004\u0020ACOS",
    "\uf201\u0004\u0020ATAN",
    "\uf201\u0004\u0020ABS",

    "\u4302\u0001\u0000=",      // Assignment token used in FOR X= (Assignment is handled by tokFOR so assignment must be suppressed)
    "\u4382\u0001\u0000#",      // TO Same Priority as , (0xnn8nSpace before but no space following)
    "\u1140\u0001\u0000PRINT",
    "\u1140\u0001\u0000INPUT",
    "\u0002\u0001\u0000:",       // : Same priority as EOS. No spc before or after
    "\u1140\u0001\u0000GELGRAB",
    "\u1140\u0001\u0000DRAWGEL",
    "\u1140\u0001\u0000SPRITEGEL",
    "\u1140\u0001\u0000SPRITEMOVE",
    "\uf201\u0004\u0020SPRITEHIT",

    "\uf201\u0004\u0020READDIR$",
    "\uf201\u0004\u0020PROPERTY$",
    "\u1140\u0001\u0000GELLOAD",
    "\uf201\u0004\u0020GELWIDTH",
    "\uf201\u0004\u0020GELHEIGHT"

    /*
    "\u1100\u4000\ua700@tokASSIGN",             // tokASSIGN
    "\uf200\u4000\ua700@tokAKEREF",             // tokMAKEREF (Same prior as functions!)
    "\u1100\u0100\u5880@PUSH_CONST_BYTE",       // PUSH_CONST_BYTE
    "\u1100\u0100\u5880@PUSH_CONST_UBYTE",      // PUSH_CONST_UBYTE
    "\u1100\u0200\u7880@tokVARIABLE",           // tokVARIABLE
    "\u1100\u0100\u5880@tokSTRING",             // tokSTRING
    "\u1100\u0100\u5880@PUSH_CONST_FLOAT",      // PUSH_CONST_FLOAT
    "\u0000\u0000\u0000@EOS"                    // EOS
 */
  };

  private static int lookupToken(String token, int matchGroup)
  {
    int tokNum = -1;
    int len = token.length();

    for (int i=0;i<tokenTable.length;i++)
    {
        int group = (tokenTable[i].charAt(0) & 0x000f);
        if ((group == matchGroup) || (matchGroup == -1))
        {
            int tokLen = tokenTable[i].length() - 3;
            if (len == tokLen)
            {
                if (token.regionMatches(true, 0, tokenTable[i], 3, len))
                {
                    tokNum = i;
                    break;
                }
            }
        }
    }

    return tokNum;
  }

  private static int lookupVariable(String token)
  {
    int variableID = -1;

    for (int i=0;i<nvars;i++)
    {
      if (token.equals(varName[i]))
      {
        variableID=i;
        break;
      }
    }

    if ((variableID == -1) && (nvars < 256))
    {
      varName[nvars] = token;

      if (token.endsWith("$"))
      {
        varType[nvars] = TYPE_STRING;
        varObject[nvars] = new String[1];
      }
      else if (token.endsWith("%"))
      {
        varType[nvars] = TYPE_INTEGER;
        varObject[nvars] = new int[1];
      }
      else
      {
        varType[nvars] = TYPE_FLOAT;
        varObject[nvars] = new int[1];
      }

      variableID = nvars++;
    }

    return variableID;
  }
  
  private static int operandStackValue[] = new int[256];
  private static Object operandStackObject[] = new Object[256];
  private static byte operandStackType[] = new byte[256];
  private static byte operandStackClass[] = new byte[256];
  private static int valueSP = -1;

  private static int operatorStack[] = new int[256];
  private static int operatorPrior[] = new int [256];
  private static int operatorSP = -1;

  private static Object controlStack[] = new Object[256];
  private static int controlSP = -1;

  private static int commaCount = 0;
  private static int argCount = 0;

  private static void PushOperand(int v, Object obj, byte t, byte c)
  {
    if (DEBUG)
    {
      System.out.println("Pushing: " +
                         "v=" + v + " " +
                         "obj=" + obj + " " +
                         "t=" + t + " " +
                         "c=" + c);
    }

    ++valueSP;
    operandStackValue[valueSP] = v;
    operandStackObject[valueSP] = obj;
    operandStackType[valueSP] = t;
    operandStackClass[valueSP] = c;
  }

  private static void PushInt(int ival)
  {
    if (DEBUG)
      System.out.println("PushInt: " + ival);
    PushOperand(ival, null, TYPE_INTEGER, CLASS_CONSTANT);
  }

  private static void PushFloat(int fval)
  {
    if (DEBUG)
      System.out.println("PushFloat: " + Float.toString(fval));
    PushOperand(fval, null, TYPE_FLOAT, CLASS_CONSTANT);
  }

  private static void PushString(String sval)
  {
    if (DEBUG)
      System.out.println("PushString: " + sval);
    PushOperand(0, sval, TYPE_STRING, CLASS_CONSTANT);
  }

  private static int PopInt()
  {
    int ival = operandStackValue[valueSP];
    poppedObject = operandStackObject[valueSP];
    poppedType = operandStackType[valueSP];
    poppedClass = operandStackClass[valueSP];

    valueSP--;

    if (poppedType == TYPE_STRING)
        throw new BasicError(BasicError.VALUE_ERROR, "String value unexpected");
        
    if (poppedClass == CLASS_VARIABLE)
    {
        int var = ival & 0x000000ff;
        int index = (ival >> 8) & 0x00ffffff;
        int[] t = (int[])varObject[var];
        ival = t[index];
    }
    
    if (poppedType == TYPE_FLOAT)
      ival = Float.ftoi(ival);
    else if (poppedType != TYPE_INTEGER)
        throw new BasicError(BasicError.VALUE_ERROR, "Integer required");

    return ival;
  }

  private static int PopFloat()
  {
    int ival = operandStackValue[valueSP];
    poppedObject = operandStackObject[valueSP];
    poppedType = operandStackType[valueSP];
    poppedClass = operandStackClass[valueSP];

    valueSP--;

    if (poppedType == TYPE_STRING)
        throw new BasicError(BasicError.VALUE_ERROR, "String value unexpected");

    if (poppedClass == CLASS_VARIABLE)
    {
        int var = ival & 0x000000ff;
        int index = (ival >> 8) & 0x00ffffff;
        int[] t = (int[])varObject[var];
        ival = t[index];
    }
    
    if (poppedType == TYPE_INTEGER)
      ival = Float.itof(ival);
    else if (poppedType != TYPE_FLOAT)
        throw new BasicError(BasicError.VALUE_ERROR, "Float required");

    return ival;
  }

  private static String PopString()
  {
    int ival = operandStackValue[valueSP];
    poppedObject = operandStackObject[valueSP];
    poppedType = operandStackType[valueSP];
    poppedClass = operandStackClass[valueSP];

    valueSP--;
    
    if (poppedType != TYPE_STRING)
        throw new BasicError(BasicError.VALUE_ERROR, "String required");

    if (poppedClass == CLASS_VARIABLE)
    {
        int var = ival & 0x000000ff;
        int index = (ival >> 8) & 0x00ffffff;
        String[] t = (String[])varObject[var];
        poppedObject = t[index];
    }
    
    return (String)poppedObject;
  }

  private static int PopType(int offsetFromTop)
  {
    return operandStackType[valueSP-offsetFromTop];
  }

  private static Object poppedObject;
  private static byte poppedType;
  private static byte poppedClass;

  /*
   * v 0..255
   * type INTEGER(0), FLOAT(1), STRING(2)
   * index
   */
  private static int PopRef()
  {
    if (valueSP == -1)
        throw new BasicError(BasicError.INTERNAL_ERROR, "Operand stack underflow");

    //int index = 0;
    int ref = operandStackValue[valueSP];
    int t = operandStackType[valueSP];
    int c = operandStackClass[valueSP];

    valueSP--;
            
    if (c != CLASS_VARIABLE)
        throw new BasicError(BasicError.INTERNAL_ERROR, "Variable expected on operand stack");
        
    if (DEBUG)
      System.out.println("PopRef(): " + ref);

    // Don't need to put type in return - its implicit by variable type

    if (t != varType[ref & 0x000000ff])
        throw new BasicError(BasicError.INTERNAL_ERROR, "Type mismatch between stack type and physical type");
    
    return ref;
    //return (index << 8) | v;
  }
    
  private static void Execute(int oper)
  {
    if (DEBUG)
    {
        System.out.print("Executing( " + oper + ") : ");
        if (oper < tokenTable.length)
            System.out.println(tokenTable[oper].substring(3));
        else
            System.out.println(oper);
        //DumpOperandStack();
    }

    switchStatement:
      switch (oper)
      {
          case tokSTOP :                            // Value stack: null
              if (DEBUG)
                  System.out.println("STOP");
              stopProgramFlag = true;
              break;

          case tokEND :                             // Value stack: null
              if (DEBUG)
                  System.out.println("END");
              exePC = exeLen;
              exeNextLinePC = exeLen;
              break;

          case tokNEW :                             // Value stack: null
              New();
              exePC = exeLen;
              exeNextLinePC = exeLen;
              break;

      case tokRUN :
        exeProg = sourceProg;
        exeLen = sourceLen;
        exePC = 0;
        exeNextLinePC = 0;
        break;

      case tokDIR :
      {
          Enumeration directoryEnumeration = support.Directory("*");

          if (directoryEnumeration != null)
          {
            while (directoryEnumeration.hasMoreElements())
            {
              support.PrintString((String)directoryEnumeration.nextElement() + "\n");
            }
          }
/*
          String[] filename = support.Directory("*");

          if (filename != null)
              for (int i=0;i<filename.length;i++)
                  if (filename[i].charAt(0) != '.')
                      support.PrintString(filename[i] + "\n");
*/
          break;
      }

      case tokREADDIR$ :
      {
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");

          String filter = PopString();

          if (filter.length() > 0)
            dirEnum = support.Directory(filter);

          if ((dirEnum != null) && (dirEnum.hasMoreElements()))
            PushString((String)dirEnum.nextElement());
          else
            PushString("");

          break;
      }

      case tokPROPERTY$ :
      {
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");

          String property = PopString();

          String value = System.getProperty(property);
          if (value != null)
            PushString(value);
          else
            PushString("");

          break;
      }
      
      case tokDEG :
        degFlag = true;
        break;

      case tokRAD :
        degFlag = false;
        break;

      case tokBYE :
          support.Bye();
          exePC = exeLen;
          exeNextLinePC = exeLen;
          break;

      case tokRETURN :
      case tokPOP :
      {
          whileLoop:
              while (controlSP >= 0)
              {
                  int controlItem = ((Integer)controlStack[controlSP--]).intValue();
                  if (controlItem == CONTROL_RETURN)
                  {
                      if (oper == tokRETURN)
                      {
                          exeNextLinePC = ((Integer)controlStack[controlSP--]).intValue();
                          exePC = ((Integer)controlStack[controlSP--]).intValue();
                          exeLen = ((Integer)controlStack[controlSP--]).intValue();
                          exeProg = (byte[])controlStack[controlSP--];
                      }
                      else
                      {
                          controlSP--;                          // exeNextLinePC
                          controlSP--;                          // exePC
                          controlSP--;                          // exeLen
                          controlSP--;                          // exeProg
                      }
                      break switchStatement;
                  }
                  else if (controlItem == CONTROL_FORLOOP)
                  {
                      controlSP--;                              // stepValue
                      controlSP--;                              // toValue
                      controlSP--;                              // controlVariable
                      controlSP--;                          // exeNextLinePC
                      controlSP--;                          // exePC
                      controlSP--;                          // exeLen
                      controlSP--;                          // exeProg
                      
                      if (oper == tokPOP)
                          break switchStatement;
                  }
                  else
                  {
                      throw new BasicError(BasicError.INTERNAL_ERROR, "Bad item on control stack");
                  }
              }
              throw new BasicError(BasicError.STACK_EMPTY, "Empty stack");
      }

      case tokGOSUB :
        {
          controlStack[++controlSP] = exeProg;
          controlStack[++controlSP] = new Integer(exeLen);
          controlStack[++controlSP] = new Integer(exePC);
          controlStack[++controlSP] = new Integer(exeNextLinePC);
          controlStack[++controlSP] = new Integer(CONTROL_RETURN);
        }
      
      case tokGOTO :
        {
          int ival = PopInt();
          exeProg = sourceProg;
          exeLen = sourceLen;
          exePC = FindLine(ival, exeProg, exeLen);
          if (exePC == -1)
            throw new BasicError(BasicError.LINE_NOT_FOUND, "Line " + ival + " not found");
          exeNextLinePC = exePC;
          break;
        }

      case tokSLEEP :
        {
          int ival = PopInt();

          if (ival > 0)
          {
            try
            {
              Thread.sleep(ival);
            }
            catch (InterruptedException e)
            {
            }
          }
          else
          {
            Thread.yield();
          }
        }
        break;

      case tokPRINT :
        {
          //if (DEBUG)
          //{
          //  System.out.print("PRINT> ");
          //  DumpOperandStack();
          //}
          
          StringBuffer sb = new StringBuffer();

          int popType = PopType(0);

          if (popType == TYPE_INTEGER)
          {
            int ival = PopInt();
            sb.insert(0, ival);
          }
          else if (popType == TYPE_FLOAT)
          {
            int fval = PopFloat();
            sb.insert(0, Float.toString(fval));
          }
          else if (popType == TYPE_STRING)
          {
            String sval = PopString();
            sb.insert(0, sval);
          }

          sb.append("\n");

          support.PrintString(sb.toString());
          
          if (DEVELOPMENT_MODE)
              System.out.println(sb.toString());
          break;
        }

      case tokREM :
          break;
          
      case tokDIM :     // Value stack: Variable Dimension
        {
            //if (DEBUG)
            //{
            //    System.out.print("DIM> ");
            //    DumpOperandStack();
            //}
            
            int ref = PopRef();
            int var = ref & 0x000000ff;
            int index = (ref >> 8) & 0x00ffffff;

            if (DEBUG)
            {
                System.out.println("var = " + var);
                System.out.println("index = " + index);
            }
            
            switch (varType[var])
            {
                case TYPE_INTEGER :
                    varObject[var] = new int[index];
                    break;
                case TYPE_FLOAT :
                    varObject[var] = new int[index];
                    break;
                case TYPE_STRING :
                    varObject[var] = new String[index];
                    break;
                default :
                    throw new BasicError(BasicError.INTERNAL_ERROR, "DIM: Bad Variable Type");
            }
          break;
        }

      case tokIF :
        {
          int result = PopInt();
          if (result == 0)
            exePC = exeNextLinePC;
          break;
        }

      case tokTHEN :
        break;
        
      case tokCOLON :
        break;

      case tokCLS :
        support.CLS();
        break;

      case tokPLOT :
        {
          int n2 = PopInt();
          int n1 = PopInt();
          support.DrawLine(n1, n2, n1, n2);
          break;
        }

      case tokDRAWLINE :
        {
          int n4 = PopInt();
          int n3 = PopInt();
          int n2 = PopInt();
          int n1 = PopInt();
          support.DrawLine(n1, n2, n3, n4);
          break;
        }

      case tokFILLRECT :
        {
          int n4 = PopInt();
          int n3 = PopInt();
          int n2 = PopInt();
          int n1 = PopInt();
          //System.out.println("n1=" + n1 + ", n2=" + n2 + ", n3=" + n3 + ", n4=" + n4);
          support.FillRect(n1, n2, n3, n4);
          break;
        }

      case tokDRAWRECT :
        {
          int n4 = PopInt();
          int n3 = PopInt();
          int n2 = PopInt();
          int n1 = PopInt();
          support.DrawRect(n1, n2, n3, n4);
          break;
        }


          case tokFILLROUNDRECT :
          {
              int n6 = PopInt();
              int n5 = PopInt();
              int n4 = PopInt();
              int n3 = PopInt();
              int n2 = PopInt();
              int n1 = PopInt();
              support.FillRoundRect(n1, n2, n3, n4, n5, n6);
              break;
          }

          case tokDRAWROUNDRECT :
          {
              int n6 = PopInt();
              int n5 = PopInt();
              int n4 = PopInt();
              int n3 = PopInt();
              int n2 = PopInt();
              int n1 = PopInt();
              support.DrawRoundRect(n1, n2, n3, n4, n5, n6);
              break;
          }

          case tokFILLARC :
          {
              int n6 = PopInt();
              int n5 = PopInt();
              int n4 = PopInt();
              int n3 = PopInt();
              int n2 = PopInt();
              int n1 = PopInt();
              support.FillArc(n1, n2, n3, n4, n5, n6);
              break;
          }
          
          case tokDRAWARC :
          {
              int n6 = PopInt();
              int n5 = PopInt();
              int n4 = PopInt();
              int n3 = PopInt();
              int n2 = PopInt();
              int n1 = PopInt();
              support.DrawArc(n1, n2, n3, n4, n5, n6);
              break;
          }

          case tokDRAWSTRING :
          {
              int n3 = PopInt();
              int n2 = PopInt();
              String n1 = PopString();
              support.DrawString(n1, n2, n3);
              break;
          }

          case tokSETCOLOR :
          {
              int n3 = PopInt();
              int n2 = PopInt();
              int n1 = PopInt();
              support.SetColor(n1, n2, n3);
              break;
          }

          case tokGELLOAD :
          {
            String resourceName = PopString();
            String gelName = PopString();
            support.GelLoad(gelName, resourceName);
            break;
          }

          case tokGELGRAB :
          {
              int h = PopInt();
              int w = PopInt();
              int y = PopInt();
              int x = PopInt();
              String gelName = PopString();
              support.GelGrab(gelName, x, y, w, h);
              break;
          }

          case tokDRAWGEL :
            {
              int y = PopInt();
              int x = PopInt();
              String gelName = PopString();
              support.DrawGel(gelName, x, y);
              break;
            }

          case tokSPRITEGEL :
          {
              String gelName = PopString();
              String spriteName = PopString();
              support.SpriteGEL(spriteName, gelName);
              break;
          }

          case tokSPRITEMOVE :
          {
              int y = PopInt();
              int x = PopInt();
              String spriteName = PopString();
              support.SpriteMove(spriteName, x, y);
              break;
          }

          case tokSPRITEHIT :
          {
            //if (DEBUG)
            //{
            //    System.out.print("SPRITEHIT$> ");
            //    DumpOperandStack();
            //}
          
            if (argCount != 2)
              throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
            String spriteName2 = PopString();
            String spriteName1 = PopString();
            PushInt(support.SpriteHit(spriteName1, spriteName2));
            break;
          }

          case tokBLIT :
          {
              int n6 = PopInt();
              int n5 = PopInt();
              int n4 = PopInt();
              int n3 = PopInt();
              int n2 = PopInt();
              int n1 = PopInt();
              support.Blit(n1, n2, n3, n4, n5, n6);
              break;
          }
 
          case tokFOR :
          {
              //if (DEBUG)
              //{
              //    System.out.print("FOR> ");
              //    DumpOperandStack();
              //}

              int stepValue;

              if (stepFlag)
              {
                  stepFlag = false;     // Reset to false first, just in case PopInt() raises an exception
                  stepValue = PopInt();
              }
              else
              {
                  stepValue = 1;
              }
              
              int toValue = PopInt();
              int initValue = PopInt();

              int ref = PopRef();
              int var = ref & 0x000000ff;
              int index = (ref >> 24) & 0x00ffffff; // Atari BASIC doesn't allow a subscript (is that general?)

              if (varType[var] != TYPE_INTEGER)
                  throw new BasicError(BasicError.INTEGER_EXPECTED, "Loop variables must be integer");

              if (index != 0)
                  throw new BasicError(BasicError.INTERNAL_ERROR, "Loop variables cannot be arrays");

              int[] tmp = (int[])varObject[var];
              tmp[0] = initValue;
              
              controlStack[++controlSP] = exeProg;
              controlStack[++controlSP] = new Integer(exeLen);
              controlStack[++controlSP] = new Integer(exePC);
              controlStack[++controlSP] = new Integer(exeNextLinePC);
              
              controlStack[++controlSP] = new Integer(var);                     // Control Variable
              controlStack[++controlSP] = new Integer(toValue);                 // Limit
              controlStack[++controlSP] = new Integer(stepValue);               // Step
              
              controlStack[++controlSP] = new Integer(CONTROL_FORLOOP);
              break;
          }
          
          case tokSTEP :
              stepFlag = true;
          case tokFOREQ :
          case tokTO :
              //if (DEBUG)
              //{
              //    System.out.print("FOREQ/TO/STEP> ");
              //    DumpOperandStack();
              //}
              break;

          case tokNEXT :
          {
              //if (DEBUG)
              //{
              //    System.out.print("NEXT> ");
              //    DumpOperandStack();
              //}
                  
              int ref = PopRef();
              int var = ref & 0x000000ff;
              int index = (ref >> 24) & 0x00ffffff; // Atari BASIC doesn't allow a subscript (is that General?)

              if (varType[var] != TYPE_INTEGER)
                  throw new BasicError(BasicError.INTEGER_EXPECTED, "Loop variables must be integer");

              if (index != 0)
                  throw new BasicError(BasicError.INTERNAL_ERROR, "Loop variables cannot be arrays");

          whileLoop:
              while (controlSP >= 0)
              {
                  int controlItem = ((Integer)controlStack[controlSP]).intValue();
                  if (controlItem == CONTROL_FORLOOP)
                  {
                      int localSP = controlSP;
                      
                      localSP--;
                      int stepValue = ((Integer)controlStack[localSP--]).intValue();
                      int toValue = ((Integer)controlStack[localSP--]).intValue();
                      int ctrlVar = ((Integer)controlStack[localSP--]).intValue();
                      
                      if (ctrlVar == var)
                      {
                          int[] tmp = (int[])varObject[ctrlVar];
                          tmp[0] = tmp[0] + stepValue;

                          if (((stepValue > 0) && (tmp[0] <= toValue)) ||
                              ((stepValue < 0) && (tmp[0] >= toValue)))
                          {
                              exeNextLinePC = ((Integer)controlStack[localSP--]).intValue();
                              exePC = ((Integer)controlStack[localSP--]).intValue();
                              exeLen = ((Integer)controlStack[localSP--]).intValue();
                              exeProg = (byte[])controlStack[localSP--];
                          }
                          else
                          {
                              controlSP = localSP;
                              controlSP--;
                              controlSP--;
                              controlSP--;
                              controlSP--;
                          }

                          break switchStatement;
                      }
                      else
                      {
                          controlSP = localSP;
                          controlSP--;
                          controlSP--;
                          controlSP--;
                          controlSP--;
                      }
                  }
              }
              throw new BasicError(BasicError.NEXT_BEFORE_FOR, "NEXT before FOR");
          }

          case tokINPUT :
          {
              //if (DEBUG)
              //{
              //    System.out.print("INPUT> ");
              //    DumpOperandStack();
              //}
              
              int ref = PopRef();
              int var = (ref & 0x000000ff);
              int index = (ref >> 8) & 0x00ffffff;
              String s = support.GetLine(PopString(), "");

              /*
               * This switch is basically the same as the one in the READ statement, it will probably
               * also be used in file GET and INPUT statements and is an ideal candiate for making into
               * a function.
               */
              
              switch (varType[var])
              {
                  case TYPE_INTEGER :
                  {
                    int ival;
                  
                      s = s.trim();
                          
                      try
                      {
                          ival = Integer.parseInt(s);
                      }
                      catch (NumberFormatException e)
                      {
                          ival = Float.fromString(s);
                          ival = Float.ftoi(ival);
                      }
                  
                      int[] obj = (int[])varObject[var];
                      obj[index] = ival;
                      break;
                  }
                  case TYPE_FLOAT :
                  {
                      s = s.trim();
                      int ival = Float.fromString(s);
                      int[] obj = (int[])varObject[var];
                      obj[index] = ival;
                      break;
                  }
                  case TYPE_STRING :
                  {
                      String[] obj = (String[])varObject[var];
                      obj[index] = s;
                      break;
                  }
                  default :
                      throw new BasicError(BasicError.INTERNAL_ERROR, "Invalid Type");
              }
              break;
          }

      case tokLOAD :
      {
          New();
          String filename = PopString();
          support.OpenFile(0, filename, true);
          DataInput dataInput = support.GetDataInputChannel(0);
          try
          {
            if (dataInput != null)
              LoadFrom(dataInput);
          }
          finally
          {
            support.CloseFile(0);
          }
          exePC = exeLen;
          exeNextLinePC = exeLen;
          break;
      }

      case tokENTER :
      {
          String filename = PopString();
          
          support.OpenFile(0, filename, true);
          DataInput dataInput = support.GetDataInputChannel(0);
          if (dataInput != null)
              Enter(dataInput);
          support.CloseFile(0);
          exePC = exeLen;
          exeNextLinePC = exeLen;
          break;
      }

      case tokLIST :
      {
          //if (DEBUG)
          //{
          //    System.out.print("LIST> ");
          //    DumpOperandStack();
          //}

          String filename = null;
          int firstLine = 0;
          int lastLine = 65535;

          if (valueSP == 0)			// List filename | List lno
          {
              if (PopType(0) == TYPE_STRING)
              {
                filename = PopString();
              }
              else
              {
                firstLine = PopInt();
                lastLine = firstLine;
              }
          }
          else if (valueSP == 1)		// List lno1, lno2
          {
              lastLine = PopInt();
              firstLine = PopInt();
          }
          else if (valueSP != -1)
              throw new BasicError(BasicError.INTERNAL_ERROR, "List: Invalid number of arguments");

          if (filename != null)
          {
            DataOutput out = null;
            support.Delete(filename);
            support.OpenFile(0, filename, false);
            DataOutput dataOutput = support.GetDataOutputChannel(0);
            List(dataOutput, firstLine, lastLine, false);
            support.CloseFile(0);
          }
          else
          {
            List(null, firstLine, lastLine, false);
          }

          break;
      }

      case tokSAVE :
      {
          String filename = PopString();
          support.Delete(filename);
          support.OpenFile(0, filename, false);
          DataOutput dataOutput = support.GetDataOutputChannel(0);
          if (dataOutput != null)
              SaveTo(dataOutput);
          support.CloseFile(0);
          break;
      }
          
      case tokDELETE :
          support.Delete(PopString());
          break;
          
      case tokEDIT :
      {
          int lno = PopInt();
          List(null, lno, lno, true);
          break;
      }
            
      case tokTRAP :
      {
          int lno = PopInt();
          if (lno != -1)
          {
              trapPC = FindLine(lno, exeProg, exeLen);
              if (trapPC == -1)
                  throw new BasicError(BasicError.LINE_NOT_FOUND, "Line " + lno + " not found");
          }
          else
          {
              trapPC = -1;
          }
          break;
      }
          
      case tokHASH :
          break;
              
      case tokOPEN :
      {
          String mode = PopString();
          String filename = PopString();
          int channel = PopInt();
          
          if (mode.regionMatches(true, 0, "OUTPUT", 0, 6))
              support.OpenFile(channel, filename, false);
          else if (mode.regionMatches(true, 0, "INPUT", 0, 5))
              support.OpenFile(channel, filename, true);
          else
              throw new BasicError(BasicError.INVALID_IO_MODE, "Invalid Open Mode");
          break;
      }
      
      case tokCLOSE :
      {
          int channel = PopInt();
          support.CloseFile(channel);
          break;
      }
      
      case tokNOTE :
      {
          int ref = PopRef();
          int var = ref & 0x000000ff;
          int index = (ref >> 8) & 0x00ffffff;
          int channel = PopInt();
          
          switch (varType[var])
          {
              case TYPE_INTEGER :
              {
                  int[] t = (int[])varObject[var];
                  t[index] = support.Note(channel);
if (DEBUG)
  System.out.println("*** NOTE = " + t[index] + " var=" + var + " index=" + index + "******************");
                  break;
              }
              
              default :
                  throw new BasicError(BasicError.INTEGER_EXPECTED, "Integer Expected");
          }
          break;
      }
          
      case tokPOINT :
      {
          int pointVal = PopInt();
          int channel = PopInt();
          support.Point(channel, pointVal);
          break;
      }
      
      case tokPUT :
      {
          int byteVal = PopInt();
          int channel = PopInt();
          support.PutByte(channel ,byteVal);
          break;
      }
          
      case tokGET :
      {
          int ref = PopRef();
          int var = ref & 0x000000ff;
          int index = (ref >> 8) & 0x00ffffff;
          int channel = PopInt();

          switch (varType[var])
          {
              case TYPE_INTEGER :
              {
                  int[] t = (int[])varObject[var];
                  t[index] = support.GetByte(channel);
                  break;
              }

              default :
                  throw new BasicError(BasicError.INTEGER_EXPECTED, "Integer Expected");
          }
          break;
      }
      
      case tokPRINTHASH :
      {
          int type = PopType(0);
          switch (type)
          {
              case TYPE_INTEGER :
              {
                  int ival = PopInt();
                  int channel = PopInt();
                  support.PutInt(channel, ival);
                  break;
              }

              case TYPE_FLOAT :
              {
                  int fval = PopFloat();
                  int channel = PopInt();
                  support.PutInt(channel, fval);
                  break;
              }

              case TYPE_STRING :
              {
                  String sval = PopString();
                  int channel = PopInt();
                  support.PutString(channel, sval);
                  break;
              }

              default :
                  throw new BasicError(BasicError.INTERNAL_ERROR, "Invalid Type");
          }
          break;
      }
      
      case tokINPUTHASH :
      {
          int ref = PopRef();
          int var = ref & 0x000000ff;
          int index = (ref >> 8) & 0x00ffffff;
          int channel = PopInt();
          switch (varType[var])
          {
              case TYPE_INTEGER :
              {
                  int[] t = (int[])varObject[var];
                  t[index] = support.GetInt(channel);
                  break;
              }

              case TYPE_FLOAT :
              {
                  int[] t = (int[])varObject[var];
                  t[index] = support.GetInt(channel);
                  break;
              }

              case TYPE_STRING :
              {
                  String[] t = (String[])varObject[var];
                  t[index] = support.GetString(channel);
                  break;
              }
              
              default :
                  throw new BasicError(BasicError.INTERNAL_ERROR, "Invalid Type");
          }
          break;
      }

      case tokDATA :
          break;
          
      case tokRESTORE :
      {
          int lno = PopInt();
          if (lno == -1)
            dataPC = 0;
          else
            dataPC = FindLine(lno, exeProg, exeLen);

          if (dataPC == -1)
            throw new BasicError(BasicError.LINE_NOT_FOUND, "Line " + lno + " not found");

          dataOffset = -1;
          dataLen = -1;
          break;
      }
      
      case tokREAD :
      {
          /*
           * prog[dataPC+0] = hi
           * prog[dataPC+1] = lo
           * prog[dataPC+2] = lineLen
           * prog[dataPC+3] = opDATA
           */

          //if (DEBUG)
          //{
          //    System.out.print("READ> ");
          //    DumpOperandStack();
          //}
          
          if (dataOffset >= dataLen)
          {
              if (dataLen != -1)
                  dataPC += (dataLen + 1);  // Don't forget to skip the EOS byte
                                    
              while (exeProg[dataPC+3] != tokDATA)
              {
                  dataPC += (int)(exeProg[dataPC+2] & 0xff);
                  if (dataPC >= exeLen)
                      throw new BasicError(BasicError.OUT_OF_DATA, "Out of Data");
              }
                                    
              dataOffset = 5;
              dataLen = dataOffset + (int)(exeProg[dataPC+4] & 0xff);
              //dataLen -= 1; // Subtract one for the EOS which terminates every line
          }
                                
          StringBuffer sb = new StringBuffer();
                                
          if (DEBUG)
              System.out.println("READ> dataOffset=" + dataOffset + ", dataLen=" + dataLen);
          
          while (dataOffset < dataLen)
          {
              char ch = (char)(exeProg[dataPC+dataOffset++] & 0xff);
              if (ch == ',')
                  break;

              sb.append(ch);
          }

          if (DEBUG)
              System.out.println("READ> \"" + sb.toString() + "\"");

          int ref = PopRef();
          int index = (ref >> 8) & 0x003fffff;
          int var = ref & 0x000000ff;
          int type = varType[var];

          String s = sb.toString();
          switch (type)
          {
              case TYPE_INTEGER :
              {
                  int ival;
                  
                  s = s.trim();
                      
                  try
                  {
                      ival = Integer.parseInt(s);
                  }
                  catch (NumberFormatException e)
                  {
                      ival = Float.fromString(s);
                      ival = Float.ftoi(ival);
                  }
                  
                  int[] obj = (int[])varObject[var];
                  obj[index] = ival;
                  break;
              }
              case TYPE_FLOAT :
              {
                  s = s.trim();
                  int ival = Float.fromString(s);
                  int[] obj = (int[])varObject[var];
                  obj[index] = ival;
                  break;
              }
              case TYPE_STRING :
              {
                  String[] obj = (String[])varObject[var];
                  obj[index] = s;
                  break;
              }
              default :
                  throw new BasicError(BasicError.INTERNAL_ERROR, "Invalid Type");
          }
          break;
      }


      case tokLBRACKET :
          if (DEBUG)
              System.out.println("LBRACKET: Setting argCount to " + (commaCount+1));
        argCount = commaCount + 1;
        commaCount = 0;
        break;

      case tokRBRACKET :
          if (DEBUG)
              System.out.println("RBRACKET: Resetting commaCount");
          commaCount = 0;
        break;
        
      case tokEOS :
          if (DEBUG)
              System.out.println("EOS: Resetting commaCount");
          commaCount = 0;
        break;

      case tokCOMMA :
        commaCount++;
        break;

      case tokEQ :
        {
          //if (DEBUG)
          //{
          //    System.out.print("EQ> ");
          //    DumpOperandStack();
          //}
          
          int type1 = PopType(0);
          int type2 = PopType(1);

          if ((type1 == TYPE_STRING) ||  (type2 == TYPE_STRING))
          {
            String right = PopString();
            String left = PopString();
            PushInt(left.compareTo(right) == 0 ? 1 : 0);
          }
          else if ((type1 == TYPE_FLOAT) || (type2 == TYPE_FLOAT))
          {
            int right = PopFloat();  
            int left = PopFloat();  
            PushInt(Float.Compare(left,right) == 0 ? 1 : 0);
          }
          else
          {
            int right = PopInt();  
            int left = PopInt();  
            PushInt(left == right ? 1 : 0);
          }
          break;
        }

      case tokNE :
        {
          int type1 = PopType(0);
          int type2 = PopType(1);

          if ((type1 == TYPE_STRING) || (type2 == TYPE_STRING))
          {
            String right = PopString();
            String left = PopString();
            PushInt(left.compareTo(right) != 0 ? 1 : 0);
          }
          else if ((type1 == TYPE_FLOAT) || (type2 == TYPE_FLOAT))
          {
            int right = PopFloat();  
            int left = PopFloat();  
            PushInt(Float.Compare(left,right) != 0 ? 1 : 0);
          }
          else
          {
            int right = PopInt();  
            int left = PopInt();  
            PushInt(left != right ? 1 : 0);
          }
          break;
        }

      case tokLT :
        {
          int type1 = PopType(0);
          int type2 = PopType(1);

          if ((type1 == TYPE_STRING) || (type2 == TYPE_STRING))
          {
            String right = PopString();
            String left = PopString();
            PushInt(left.compareTo(right) < 0 ? 1 : 0);
          }
          else if ((type1 == TYPE_FLOAT) || (type2 == TYPE_FLOAT))
          {
            int right = PopFloat();  
            int left = PopFloat();  
            PushInt(Float.Compare(left,right) < 0 ? 1 : 0);
          }
          else
          {
            int right = PopInt();  
            int left = PopInt();  
            PushInt(left < right ? 1 : 0);
          }
          break;
        }

      case tokLE :
        {
          int type1 = PopType(0);
          int type2 = PopType(1);

          if ((type1 == TYPE_STRING) || (type2 == TYPE_STRING))
          {
            String right = PopString();
            String left = PopString();
            PushInt(left.compareTo(right) <= 0 ? 1 : 0);
          }
          else if ((type1 == TYPE_FLOAT) || (type2 == TYPE_FLOAT))
          {
            int right = PopFloat();  
            int left = PopFloat();  
            PushInt(Float.Compare(left,right) <= 0 ? 1 : 0);
          }
          else
          {
            int right = PopInt();  
            int left = PopInt();  
            PushInt(left <= right ? 1 : 0);
          }
          break;
        }

      case tokGT :
        {
          int type1 = PopType(0);
          int type2 = PopType(1);

          if ((type1 == TYPE_STRING) || (type2 == TYPE_STRING))
          {
            String right = PopString();
            String left = PopString();
            PushInt(left.compareTo(right) > 0 ? 1 : 0);
          }
          else if ((type1 == TYPE_FLOAT) || (type2 == TYPE_FLOAT))
          {
            int right = PopFloat();  
            int left = PopFloat();  
            PushInt(Float.Compare(left,right) > 0 ? 1 : 0);
          }
          else
          {
            int right = PopInt();  
            int left = PopInt();  
            PushInt(left > right ? 1 : 0);
          }
          break;
        }

      case tokGE :
        {
          int type1 = PopType(0);
          int type2 = PopType(1);

          if ((type1 == TYPE_STRING) || (type2 == TYPE_STRING))
          {
            String right = PopString();
            String left = PopString();
            PushInt(left.compareTo(right) >= 0 ? 1 : 0);
          }
          else if ((type1 == TYPE_FLOAT) || (type2 == TYPE_FLOAT))
          {
            int right = PopFloat();  
            int left = PopFloat();  
            PushInt(Float.Compare(left,right) >= 0 ? 1 : 0);
          }
          else
          {
            int right = PopInt();  
            int left = PopInt();  
            PushInt(left >= right ? 1 : 0);
          }
          break;
        }

      case tokADD :
        {
          int type1 = PopType(0);
          int type2 = PopType(1);

          if ((type1 == TYPE_STRING) || (type2 == TYPE_STRING))
          {
            String right = PopString();
            String left = PopString();
            PushString(left + right);
          }
          else if ((type1 == TYPE_FLOAT) || (type2 == TYPE_FLOAT))
          {
            int right = PopFloat();  
            int left = PopFloat();  
            PushFloat(Float.Add(left,right));
          }
          else
          {
            int right = PopInt();  
            int left = PopInt();  
            PushInt(left + right);
          }
          break;
        }

      case tokSUB :
        {
            //if (DEBUG)
            //{
            //    System.out.print("SUB> ");
            //    DumpOperandStack();
            //}
            
          int type1 = PopType(0);
          int type2 = PopType(1);

          if ((type1 == TYPE_FLOAT) || (type2 == TYPE_FLOAT))
          {
            int right = PopFloat();  
            int left = PopFloat();  
            PushFloat(Float.Subtract(left,right));
          }
          else
          {
            int right = PopInt();  
            int left = PopInt();  
            PushInt(left - right);
          }
          break;
        }

      case tokMULT :
        {
          int type1 = PopType(0);
          int type2 = PopType(1);

          if ((type1 == TYPE_FLOAT) || (type2 == TYPE_FLOAT))
          {
            int right = PopFloat();  
            int left = PopFloat();  
            PushFloat(Float.Multiply(left,right));
          }
          else
          {
            int right = PopInt();  
            int left = PopInt();  
            PushInt(left * right);
          }
          break;
        }

      case tokDIV :
        {
          int type1 = PopType(0);
          int type2 = PopType(1);

          if ((type1 == TYPE_FLOAT) || (type2 == TYPE_FLOAT))
          {
            int right = PopFloat();  
            int left = PopFloat();  
            PushFloat(Float.Divide(left,right));
          }
          else
          {
            int right = PopInt();  
            int left = PopInt();  
            PushInt(left / right);
          }
          break;
        }
      
      case tokPOWER :
        {
          int right = PopFloat();
          int left = PopFloat();
          PushFloat(Float.pow(left, right));
          break;
        }

      case tokUMINUS :
        {
          int type1 = PopType(0);

          if (type1 == TYPE_INTEGER)
          {
            PushInt(-PopInt());
          }
          else
          {
            PushFloat(Float.Negate(PopFloat()));
          }
          break;
        }

          case tokBITAND :
        {
          int right = PopInt();  
          int left = PopInt();  
          PushInt(left & right);
          break;
        }

      case tokBITOR :
        {
          int right = PopInt();  
          int left = PopInt();  
          PushInt(left | right);
          break;
        }

      case tokBITXOR :
        {
          int right = PopInt();  
          int left = PopInt();  
          PushInt(left ^ right);
          break;
        }

        case tokLOGNOT :
        {
          PushInt(PopInt() == 0 ? 1 : 0);
          break;
        }

          case tokLOGAND :
        {
          int right = PopInt();  
          int left = PopInt();  
          PushInt((left != 0) && (right != 0) ? 1 : 0);
          break;
        }

      case tokLOGOR :
        {
          int right = PopInt();  
          int left = PopInt();  
          PushInt((left != 0) || (right != 0) ? 1 : 0);
          break;
        }

      case tokSCREENWIDTH :
          //if (DEBUG)
          //{
          //    System.out.print("SCREENWIDTH> ");
          //    DumpOperandStack();
          //}
      
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PopInt(); // Dummy argument
          PushInt(support.ScreenWidth());
          break;

      case tokSCREENHEIGHT :
          //if (DEBUG)
          //{
          //    System.out.print("SCREENHEIGHT> ");
          //    DumpOperandStack();
          //}
          
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PopInt(); // Dummy argument
          PushInt(support.ScreenHeight());
          break;

      case tokGELWIDTH :
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PushInt(support.GelWidth(PopString()));
          break;

      case tokGELHEIGHT :
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PushInt(support.GelHeight(PopString()));
          break;

      case tokISCOLOR :
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PopInt(); // Dummy argument
          PushInt(support.isColor());
          break;

      case tokNUMCOLORS :
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PopInt(); // Dummy argument
          PushInt(support.NumColors());
          break;

      case tokSTRINGWIDTH :
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PushInt(support.StringWidth(PopString()));
          break;

      case tokSTRINGHEIGHT :
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PushInt(support.StringHeight(PopString()));
          break;

      case tokLEFT$ :
      {
          //if (DEBUG)
          //{
          //    System.out.print("LEFT$> ");
          //    DumpOperandStack();
          //}
          
          if (argCount != 2)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int nbytes = PopInt();
          String str = PopString();
          PushString(str.substring(0, nbytes));
          break;
      }
      
      case tokMID$ :
      {
          //if (DEBUG)
          //{
          //    System.out.print("MID$> ");
          //    DumpOperandStack();
          //}
          
          if (argCount != 3)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int nbytes = PopInt();
          int start = PopInt() - 1;     // Basic Strings Index from 1 but Java strings are from 0
          String str = PopString();
          PushString(str.substring(start, start+nbytes));
          break;
      }
      
      case tokRIGHT$ :
      {
          //if (DEBUG)
          //{
          //    System.out.print("RIGHT$> ");
          //    DumpOperandStack();
          //}

          if (argCount != 2)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int nbytes = PopInt();
          String str = PopString();
          PushString(str.substring(str.length()-nbytes));
          break;
      }

      case tokCHR$ :
      {
          //if (DEBUG)
          //{
          //    System.out.print("CHR$> ");
          //    DumpOperandStack();
          //}

          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          char[] chArray = new char[1];
          int ival = PopInt();
          chArray[0] = (char)ival;
          PushString(new String(chArray));
          break;
      }
      
      case tokSTR$ :
      {
          //System.out.println("STR$: argCount=" + argCount);
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          if (PopType(0) == TYPE_FLOAT)
              PushString(Float.toString(PopFloat()));
          else
              PushString(Integer.toString(PopInt()));
          break;
      }
      
      case tokLEN :
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PushInt(PopString().length());
          break;
          
      case tokASC:
          //System.out.println("tokASC, argCount=" + argCount);
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PushInt(PopString().charAt(0));
          break;
          
      case tokVAL:
      {
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          String str = PopString();
          
          str = str.trim();
          
          try
          {
              PushInt(Integer.parseInt(str));
          }
          catch (NumberFormatException e)
          {
              PushFloat(Float.fromString(str));
          }
          break;
      }

      case tokUP :
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PopInt(); // Dummy argument
          PushInt(support.Up());
          break;

      case tokDOWN :
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PopInt(); // Dummy argument
          PushInt(support.Down());
          break;

      case tokLEFT :
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PopInt(); // Dummy argument
          PushInt(support.Left());
          break;

      case tokRIGHT :
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PopInt(); // Dummy argument
          PushInt(support.Right());
          break;

      case tokFIRE :
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PopInt(); // Dummy argument
          PushInt(support.Fire());
          break;

      case tokGAMEA :
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PopInt(); // Dummy argument
          PushInt(support.GameA());
          break;

      case tokGAMEB :
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PopInt(); // Dummy argument
          PushInt(support.GameB());
          break;

      case tokGAMEC :
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PopInt(); // Dummy argument
          PushInt(support.GameC());
          break;

      case tokGAMED :
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PopInt(); // Dummy argument
          PushInt(support.GameD());
          break;

      case tokDAYS :
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PopInt(); // Dummy argument
          PushInt((int)(System.currentTimeMillis() / MS_PER_DAY));
          break;
          
      case tokMILLISECONDS :
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PopInt(); // Dummy argument
          PushInt((int)(System.currentTimeMillis() % MS_PER_DAY));
          break;
          
      case tokYEAR :
      {
          if (argCount != 2)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int n2 = PopInt();
          int n1 = PopInt();
          PushInt(support.Year(new Date((long)n1 * MS_PER_DAY + (long)n2)));
          break;
      }
      
      case tokMONTH :
      {
          if (argCount != 2)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int n2 = PopInt();
          int n1 = PopInt();
          PushInt(support.Month(new Date((long)n1 * MS_PER_DAY + (long)n2)));
          break;
      }
      
      case tokDAY :
      {
          if (argCount != 2)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int n2 = PopInt();
          int n1 = PopInt();
          PushInt(support.Day(new Date((long)n1 * MS_PER_DAY + (long)n2)));
          break;
      }
      
      case tokHOUR :
      {
          if (argCount != 2)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int n2 = PopInt();
          int n1 = PopInt();
          PushInt(support.Hour(new Date((long)n1 * MS_PER_DAY + (long)n2)));
          break;
      }
      
      case tokMINUTE :
      {
          if (argCount != 2)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int n2 = PopInt();
          int n1 = PopInt();
          PushInt(support.Minute(new Date((long)n1 * MS_PER_DAY + (long)n2)));
          break;
      }
      
      case tokSECOND :
      {
          if (argCount != 2)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int n2 = PopInt();
          int n1 = PopInt();
          PushInt(support.Second(new Date((long)n1 * MS_PER_DAY + (long)n2)));
          break;
      }
      
      case tokMILLISECOND :
      {
          if (argCount != 2)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int n2 = PopInt();
          int n1 = PopInt();
          PushInt(support.Millisecond(new Date((long)n1 * MS_PER_DAY + (long)n2)));
          break;
      }
          
      case tokRND :
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PopInt(); // Dummy argument
          PushInt(random.nextInt());
          break;
          
      case tokERR :
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PopInt(); // Dummy argument
          PushInt(lastError);
          break;
          
/*
 * FRE(0) - sourceSize-sourceLen
 * FRE(1) - sourceLen
 * FRE(2) - sourceSize
 * FRE(3) - runtime.freeMemory()
 * FRE(4) - runtime.totalMemory()
 */
      case tokFRE :
      {
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int ival = PopInt(); // Dummy argument
          if (ival == 0)
            PushInt(sourceSize-sourceLen);
          else if (ival == 1)
            PushInt(sourceLen);
          else if (ival == 2)
            PushInt(sourceSize);
          else if (ival == 3)
            PushInt((int)runtime.freeMemory());
          else
            PushInt((int)runtime.totalMemory());
          break;
      }
          
      case tokMOD :
      {
          if (argCount != 2)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int right = PopInt();
          int left = PopInt();
          PushInt(left % right);
          break;
      }
      
      case tokEDITFORM :
      {
            //if (DEBUG)
            //{
            //    System.out.print("EDITFORM> ");
            //    DumpOperandStack();
            //}
            
          if (argCount != 7)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int mode = PopInt();
          int maxLen = PopInt();
          int ref = PopRef();
          int var = ref & 0x000000ff;
          int index = (ref >> 8) & 0x00ffffff;
          String[] stringArray = (String[])varObject[var];
          String defaultText = stringArray[index];
          String label = PopString();
          String cancelText = PopString();
          String proceedText = PopString();          
          String formTitle = PopString();
          int res = -1;
          
          String text = support.EditForm(formTitle, proceedText, cancelText, label, defaultText, maxLen, mode);
          if (text != null)
          {
              stringArray[index] = text;
              res = text.length();
          }

          PushInt(res);
          break;
      }                

      case tokGAUGEFORM :
      {
          if (argCount != 7)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int mode = PopInt();
          int initialValue = PopInt();
          int maxValue = PopInt();
          String label = PopString();
          String cancelText = PopString();
          String proceedText = PopString();
          String formTitle = PopString();
          PushInt(support.GaugeForm(formTitle, proceedText, cancelText, label, maxValue, initialValue, mode));
          break;
      }
      
      case tokCHOICEFORM :
      {
          if (argCount != 6)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int mode = PopInt();
          int ref = PopRef();
          String[] stringArray = (String[])varObject[ref & 0x000000ff];
          String label = PopString();
          String cancelText = PopString();
          String proceedText = PopString();
          String formTitle = PopString();

          PushInt(support.ChoiceForm(formTitle, proceedText, cancelText, label, stringArray, mode));
          break;
      }

      case tokDATEFORM :
      {
          if (argCount != 6)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int mode = PopInt();
          int ref = PopRef();
          int var = ref & 0x000000ff;
          int index = (ref >> 8) & 0x00ffffff;
          int[] intArray = (int[])varObject[var & 0x000000ff];
          String label = PopString();
          String cancelText = PopString();
          String proceedText = PopString();
          String formTitle = PopString();

          if (intArray.length >= 2)
          {
              Date date = null;
              int res = -1;
              
              if ((intArray[0] != -1) || (intArray[1] != -1))
                  date = new Date((long)intArray[0] * MS_PER_DAY + (long)intArray[1]);

              date = support.DateForm(formTitle, proceedText, cancelText, label, date, mode);
              if (date != null)
              {
                  long ms = date.getTime();
                                        
                  intArray[0] = (int)(ms / MS_PER_DAY);
                  intArray[1] = (int)(ms % MS_PER_DAY);
                  res = 1;
              }
              
              PushInt(res);
          }
          else
          {
              throw new BasicError(BasicError.INTEGER_ARRAY_EXPECTED, "Integer Array Expected"); 
              // throw new BasicError(BasicError.ARRAY_BOUNDS, iName[id6] + "(" + 1 + ") out of bounds"); 
          }
          break;
      }
      
      case tokMESSAGEFORM :
      {
          if (argCount != 5)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          String msg = PopString();
          String label = PopString();
          String cancelText = PopString();
          String proceedText = PopString();
          String formTitle = PopString();
          PushInt(support.MessageForm(formTitle, proceedText, cancelText, label, msg));
          break;
      }

      case tokLOG :
        {
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PushFloat(Float.log(PopFloat()));
          break;
        }

      case tokEXP :
        {
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PushFloat(Float.exp(PopFloat()));
          break;
        }

      case tokSQR :
        {
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          PushFloat(Float.sqrt(PopFloat()));
          break;
        }

      case tokSIN :
        {
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int fval = PopFloat();
          PushFloat(degFlag ? Float.sind(fval) : Float.sin(fval));
          break;
        }

      case tokCOS :
        {
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int fval = PopFloat();
          PushFloat(degFlag ? Float.cosd(fval) : Float.cos(fval));
          break;
        }

      case tokTAN :
        {
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int fval = PopFloat();
          PushFloat(degFlag ? Float.tand(fval) : Float.tan(fval));
          break;
        }

      case tokASIN :
        {
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int fval = PopFloat();
          PushFloat(degFlag ? Float.asind(fval) : Float.asin(fval));
          break;
        }

      case tokACOS :
        {
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int fval = PopFloat();
          PushFloat(degFlag ? Float.acosd(fval) : Float.acos(fval));
          break;
        }

      case tokATAN :
        {
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");
          int fval = PopFloat();
          PushFloat(degFlag ? Float.atand(fval) : Float.atan(fval));
          break;
        }

      case tokABS :
        {
          if (argCount != 1)
            throw new BasicError(BasicError.INCORRECT_NUMBER_OF_ARGUMENTS, "Incorrect number of arguments");

          if (PopType(0) == TYPE_FLOAT)
          {
              int fval = PopFloat();
              if (Float.Compare(fval, Float.ZERO) == -1)
                  fval = Float.Negate(fval);
              PushFloat(fval);
          }
          else
          {
              PushInt(Math.abs(PopInt()));
          }
          break;
      }

          case tokMAKEREF :
          {
              //if (DEBUG)
              //{
              //    System.out.print("MAKEREF> ");
              //    DumpOperandStack();
              //}
          
              int index = PopInt();
              int var = PopRef();
              int ref = (index << 8) | var;
              PushOperand(ref, null, varType[var], CLASS_VARIABLE);
              break;
          }
                    
        case tokASSIGN :	// Value Stack: Variable [index] Value
        {
            //if (DEBUG)
            //{
            //    System.out.print("ASSIGN> ");
            //    DumpOperandStack();
            //}

          int destType = PopType(1);

          //if ((varType[ivar] == TYPE_INTEGER) ||
          //    (varType[ivar] == TYPE_FLOAT))
          if (destType == TYPE_INTEGER)
          {
            int ival = PopInt();
            int ref = PopRef();
            int var = ref & 0x000000ff;
            int index = (ref >> 8) & 0x00ffffff;

            if (DEBUG)
              System.out.println("Assigning " + ival + " to var " + varName[var] + "[" + index + "]");
            
            int[] t = (int[])varObject[var];
            t[index] = ival;
          }
          else if (destType == TYPE_FLOAT)
          {
            int fval = PopFloat();
            int ref = PopRef();
            int var = ref & 0x000000ff;
            int index = (ref >> 8) & 0x00ffffff;

            if (DEBUG)
              System.out.println("Assigning " + Float.toString(fval) + " to var " + varName[var] + "[" + index + "]");

            int[] t = (int[])varObject[var];
            t[index] = fval;
          }
          else if (destType == TYPE_STRING)
          {
            String sval = PopString();
            int ref = PopRef();
            int var = ref & 0x000000ff;
            int index = (ref >> 8) & 0x00ffffff;

            if (DEBUG)
              System.out.println("Assigning \"" + sval + "\" to var " + varName[var] + "[" + index + "]");

            String[] t = (String[])varObject[var];
            t[index] = sval;
          }
          else
          {
              throw new BasicError(BasicError.INTERNAL_ERROR, "Bad Variable Type");
          }

          break;
        }

      default :
          throw new BasicError(BasicError.INTERNAL_ERROR, "Bad operation: " + oper);
    }
  }

  /*
   * When should tokEQ be replaced with tokASSIGN?
   *
   * 1. If TOP of operatorStack contains
   */
  
  private static String DumpOperatorStack()
  {
    StringBuffer sb = new StringBuffer();

    for (int i=0;i<=operatorSP;i++)
    {
      if (i != 0)
        sb.append(" ");
          
      int operator = operatorStack[i];
      int priority = operatorPrior[i];

      if (operator < tokenTable.length)
        sb.append(tokenTable[operator].substring(3));
      else
        sb.append(operator);
          
      sb.append("(" + priority + ")");
    }

    return sb.toString();
  }
  
  private static String DumpOperandStack()
  {
    StringBuffer sb = new StringBuffer();

    for (int i=0;i<=valueSP;i++)
    {
      if (i != 0)
        sb.append(" ");

      int val = operandStackValue[i];
      if (operandStackClass[i] == CLASS_VARIABLE)
      {
        int var = val & 0x000000ff;
        int index = (val >> 8) & 0x00ffffff;
              
        sb.append(varName[var]);
        if (index > 0)
          sb.append("[" + index + "]");
      }
      else if (operandStackType[i] == TYPE_STRING)
      {
        String str = (String)operandStackObject[i];
        sb.append("\"" + str + "\"");
      }
      else if (operandStackType[i] == TYPE_FLOAT)
      {
        sb.append(Float.toString(val));
      }
      else if (operandStackType[i] == TYPE_INTEGER)
      {
        sb.append(val);
      }
      else
      {
        sb.append(val);
      }
    }

    return sb.toString();
  }

  private static void AddOperator(int oper, int onStack, int offStack)
  {
    if (DEBUG)
    {
        System.out.print("Adding: ");

        if (oper < tokenTable.length)
            System.out.print(tokenTable[oper].substring(3));
        else
            System.out.print(oper);
 
        System.out.println(" onStack=" + onStack + " " +
                           "offStack=" + offStack);
    }
    
    if (onStack != -1)
    {
        // boolean first = true;
        boolean leftBracketPopped = false;
        
      while ((operatorSP >= 0) && (operatorPrior[operatorSP] >= onStack))
      {
          /*
          if (first == true)
          {
              System.out.print("  Before> "); DumpOperatorStack();
              System.out.print("  Before> "); DumpOperandStack();
              first = false;
          }
          */
          
          int thisOper = operatorStack[operatorSP];
          if (thisOper == tokLBRACKET)
          {
              if (leftBracketPopped)                // Make sure we can't pop two left brackets - a right close one left bracket!
                  break;
              else
                  leftBracketPopped = true;
          }
          else if (leftBracketPopped)               // If we've popped off a left bracket
          {
              if (operatorPrior[operatorSP] != 2)   // then make sure we can only pop a function
                  break;
          }
          
          Execute(operatorStack[operatorSP--]);
      }
        
        /*
        if (!first)
        {
            System.out.print("  After> "); DumpOperatorStack();
            System.out.print("  After> ");DumpOperandStack();
        }
         */
    }

    if ((oper != tokEOS) && (oper != tokCOLON)) //  && (oper != tokTHEN))
    {
        operatorSP++;
      operatorStack[operatorSP] = oper;
      operatorPrior[operatorSP] = offStack;
    }

    //if (DEBUG)
    //{
    //    DumpOperatorStack();
    //    DumpOperandStack();
    //}
  }

  /*
   * An LValue is an IDENTIFIER followed by an optional index
   */
  
  static boolean parseLValue()
  {
      boolean validFlag = false;
      boolean arrayFlag = false;
      
      String token = GetToken();
      if (token != null)
      {
          char ch = token.charAt(0);
          
          if ((ch >= 'A') && (ch <= 'Z'))
          {
              int toknum = lookupToken(token, 1);
              if (toknum == -1)
              {
                  code[codeLen++] = (byte)tokVARIABLE;
                  code[codeLen++] = (byte)lookupVariable(token);
                  
                  token = GetToken();
                  if ((token != null) && (token.compareTo("(") == 0))
                  {
                    code[codeLen++] = (byte)tokMAKEREF;
                    
                    if (DEBUG)
                      System.out.println("parseLValue is calling parseRValue");

                    code[codeLen++] = (byte)tokLBRACKET;
                    parseRValue(true);
                    
                    if (DEBUG)
                        System.out.println("returned from parseRValue");

                    token = GetToken();
                    
                    if (DEBUG)
                        System.out.println("token = " + token);
                    
                      if ((token != null) && (token.compareTo(")") == 0))
                      {
                          code[codeLen++] = (byte)tokRBRACKET;
                          arrayFlag = true;
                          validFlag = true;
                      }
                  }
                  else
                  {
                      PutToken(token);
                      validFlag = true;
                  }
              }
          }
      }

      if (!validFlag)
          throw new BasicError(BasicError.LVALUE_EXPECTED, "LVALUE Expected");
      
      return arrayFlag;
  }

  /*
   * Word 1
   *   Nibble 3
   *     Go on stack priority
   *   Nibble 2
   *     Come off stack priority
   *
   * Word 2
   *   Nibble 2
   *     Bit 8      End of Expression   0x100
   *   Nibble 1
   *     Bit 7      Unary Operator      0x080
   *     Bit 6      Binary Operator     0x040
   *     Bit 5      Left Bracket        0x020
   *     Bit 4      Right Bracket       0x010
   *   Nibble 0
   *     Bit 3       Comma              0x008
   *     Bit 2       Function           0x004
   *     Bit 1       Variable/LValue    0x002
   *     Bit 0       Constant           0x001
   *
   * Current State  Next States
   * START          UNARY | LBRACKET | FUNCTION | VARIABLE | CONSTANT   %1010 0111 0000 0000    0xa700
   * UNARY          LBRACKET | FUNCTION | VARIABLE | CONSTANT           %0010 0111 0000 0000    0x2700
   * BINARY         UNARY | LBRACKET | FUNCTION | VARIABLE | CONSTANT   %1010 0111 0000 0000    0xa700
   * LBRACKET       UNARY | LBRACKET | FUNCTION | VARIABLE | CONSTANT   %1010 0111 0000 0000    0xa700
   * RBRACKET       BINARY | RBRACKET | COMMA | EOS                     %0101 1000 1000 0000    0x5880
   * COMMA          UNARY | LBRACKET | FUNCTION | VARIABLE | CONSTANT   %1010 0111 0000 0000    0xa700
   * FUNCTION       LBRACKET                                            %0010 0000 0000 0000    0x2000
   * VARIABLE       BINARY | LBRACKET | RBRACKET | COMMA | EOS          %0111 1000 1000 0000    0x7880
   * CONSTANT       BINARY | RBRACKET | COMMA | EOS                     %0101 1000 1000 0000    0x5880
   */

  /*
   * An RValue is an Expression
   */
  
    private static final int EXPR_END = 0x100;
    private static final int EXPR_UNARY = 0x80;
    private static final int EXPR_BINARY = 0x40;
    private static final int EXPR_LBRACKET = 0x20;
    private static final int EXPR_RBRACKET = 0x10;
    private static final int EXPR_COMMA = 0x08;
    private static final int EXPR_FUNCTION = 0x04;
    private static final int EXPR_LVALUE = 0x02;
    private static final int EXPR_CONSTANT = 0x01;
    

    static void parseRValue(boolean commaValid)
    {
        //int currentTokenType = 0x0000;
        int validTokenTypes = EXPR_UNARY | EXPR_LBRACKET | EXPR_FUNCTION | EXPR_LVALUE | EXPR_CONSTANT;
        int bracketLevel = 0;
        boolean[] commaValidStack = new boolean[16];
        
        String token;
        
      whileLoop:
        while ((token = GetToken()) != null)
        {
            if (DEBUG)
                System.out.println("parseRValue: token=" + token);

            char ch = token.charAt(0);
            
            if (ch == '\"')
            {
                if ((EXPR_CONSTANT & validTokenTypes) == 0x0000)
                {
                    PutToken(token);
                    break whileLoop;
                }
                
                code[codeLen++] = (byte)tokSTRING;
                int len = token.length() - 2;
                code[codeLen++] = (byte)len;
                for (int i=0;i<len;i++)
                    code[codeLen++] = (byte)token.charAt(1+i);

                validTokenTypes = EXPR_BINARY | EXPR_RBRACKET | EXPR_COMMA | EXPR_END;
            }
            else if ((ch >= '0') && (ch <= '9'))
            {
                if ((EXPR_CONSTANT & validTokenTypes) == 0x0000)
                {
                    PutToken(token);
                    break whileLoop;
                }
                
                try
                {
                    int tokval = Integer.parseInt(token);

                    if ((tokval >= -128) && (tokval < 128))
                    {
                        code[codeLen++] = (byte)tokBYTE;
                        code[codeLen++] = (byte)tokval;
                    }
                    else if ((tokval >=0) && (tokval < 256))
                    {
                        code[codeLen++] = (byte)tokUBYTE;
                        code[codeLen++] = (byte)tokval;
                    }
                    else if ((tokval >= 0) && (tokval < 65536))
                    {
                        code[codeLen++] = (byte)tokUWORD;
                        code[codeLen++] = (byte)((tokval >> 8) & 0xff);
                        code[codeLen++] = (byte)(tokval & 0xff);
                    }
                    else
                    {
                        code[codeLen++] = (byte)tokINTEGER;
                        code[codeLen++] = (byte)((tokval >> 24) & 0xff);
                        code[codeLen++] = (byte)((tokval >> 16) & 0xff);
                        code[codeLen++] = (byte)((tokval >> 8) & 0xff);
                        code[codeLen++] = (byte)(tokval & 0xff);
                    }
                }
                catch (NumberFormatException e)
                {
                    try
                    {
                        int tokval = Float.fromString(token);
                        code[codeLen++] = (byte)tokFLOAT;
                        code[codeLen++] = (byte)((tokval >> 24) & 0xff);
                        code[codeLen++] = (byte)((tokval >> 16) & 0xff);
                        code[codeLen++] = (byte)((tokval >> 8) & 0xff);
                        code[codeLen++] = (byte)(tokval & 0xff);
                    }
                    catch (Exception e2)
                    {
                        throw new BasicError(BasicError.VALUE_ERROR, "Bad Constant2: " + token);
                    }
                }

                validTokenTypes = EXPR_BINARY | EXPR_RBRACKET | EXPR_COMMA | EXPR_END;
            }
            else
            {
                int toknum = lookupToken(token, 1);
                if (toknum != -1)
                {
                    ch = token.charAt(0);
                    if (ch == '(')
                    {
                        commaValidStack[bracketLevel++] = commaValid;
                        commaValid = true;
                    }
                    else if (ch == ')')
                    {
                        if (bracketLevel == 0)
                        {
                            PutToken(token);
                            break whileLoop;
                        }
                        
                        commaValid = commaValidStack[--bracketLevel];
                    }
                    else if (!commaValid && (ch == ','))    // Is the comma Valid
                    {
                        PutToken(token);                    // No put comma back into input token list
                        break whileLoop;                    // and terminate
                    }
                    else if (ch == '-') // Perhaps it should be a Unary Minus
                    {
                        if ((validTokenTypes & EXPR_UNARY) != 0x0000)
                        {
                            if (DEBUG)
                                System.out.println("Changed to Unary Minus");
                            toknum++;   // Unary is valid so change to unary minus
                        }
                        //else
                        //{
                        //    if (DEBUG)
                        //        System.out.println("Left as Binary Minus");
                        //}
                    }
                    
                    int currentTokenType = tokenTable[toknum].charAt(1);
                    
                    if (DEBUG)
                    {
                        System.out.println("parseRValue: currentTokenType=" + Integer.toHexString(currentTokenType));
                        System.out.println("parseRValue: validTokenTypes=" + Integer.toHexString(validTokenTypes));
                    }
                    
                    if ((currentTokenType & validTokenTypes) == 0x0000)
                    {
                        PutToken(token);
                        break whileLoop;
                    }

                    code[codeLen++] = (byte)toknum;
                    
                    validTokenTypes = tokenTable[toknum].charAt(2);
                }
                else if ((ch >= 'A') && (ch <= 'Z'))
                {
                    PutToken(token);    // Replace token for parseLValue() or further processing if not valid here

                    if ((EXPR_LVALUE & validTokenTypes) == 0x0000)
                        break whileLoop;

                    /*
                     * STR$(A%(14))
                     *
                     * STR$ ( A% ( 14 ) tokARRAY_READ )
                     *
                     */
                    boolean arrayFlag = parseLValue();

                    validTokenTypes = EXPR_BINARY | EXPR_RBRACKET | EXPR_COMMA | EXPR_END;
                }
                else
                {
                    PutToken(token);    // Replace token - its not valid here!
                    break whileLoop;
                }
            }
        }
        
        if (bracketLevel != 0)
            throw new BasicError(BasicError.PARENTHESIS_NESTING_ERROR,"Parenthesis Nesting Error");
        
        if ((EXPR_END & validTokenTypes) == 0x0000)
            throw new BasicError(BasicError.EXPRESSION_INCOMPLETE, "End of Expression not Expected");
    }

    static void parseCommaList(String commaList)
    {
        for (int i=0;i<commaList.length();i++)
        {
            char ch = commaList.charAt(i);
            
            switch (ch)
            {
                case '#' :
                {
                    String token = GetToken();
                    if (token != null)
                    {
                        if (token.compareTo("#") == 0)
                            code[codeLen++] = (byte)tokHASH;
                        else
                            throw new BasicError(BasicError.HASH_EXPECTED, "# expected");
                    }
                    else
                    {
                        throw new BasicError(BasicError.HASH_EXPECTED, "# expected");
                    }
                    break;
                }
                
                case ',' :
                {
                    String token = GetToken();
                    if (token != null)
                    {
                        if (token.compareTo(",") == 0)
                            code[codeLen++] = (byte)tokCOMMA;
                        else
                            throw new BasicError(BasicError.COMMA_EXPECTED, "Comma expected");
                    }
                    else
                    {
                        throw new BasicError(BasicError.COMMA_EXPECTED, "Comma expected");
                    }
                    break;
                }
                
                case 'R' :
                    parseRValue(false); // Don't parse commas at outer bracket level
                    break;
                    
                case 'L' :
                    parseLValue();
                    break;
                    
                default :
                    throw new BasicError(BasicError.INTERNAL_ERROR, "Bad item in comma list");
            }
        }
    }
    
    static void parseStatement()
    {
        String token;
        boolean statementRequired = false;  // A statement is only required following an IF or COLON
        
        while ((token = GetToken()) != null)
        {
            statementRequired = false;
            
            int keyword = lookupToken(token, 0);
            if (keyword != -1)
            {
                if (DEBUG)
                    System.out.println("Keyword: " + tokenTable[keyword].substring(3));
                
                if (keyword == tokPRINT)
                {
                    token = GetToken();
                    if (token != null)
                    {
                        if (token.compareTo("#") == 0)
                            keyword = tokPRINTHASH;
                        PutToken(token);
                    }
                }
                else if (keyword == tokINPUT)
                {
                    token = GetToken();
                    if (token != null)
                    {
                        if (token.compareTo("#") == 0)
                            keyword = tokINPUTHASH;
                        PutToken(token);
                    }
                }
                
                code[codeLen++] = (byte)keyword;

                if ((keyword == tokDIM) || (keyword == tokREAD) || (keyword == tokNEXT))
                    parseLValue();
                else if (keyword == tokOPEN)
                {
                    parseCommaList("#R,R,R");
                }
                else if (keyword == tokCLOSE)
                {
                    parseCommaList("#R");
                }
                else if ((keyword == tokPOINT) || (keyword == tokPRINTHASH) || (keyword == tokPUT))
                {
                    parseCommaList("#R,R");
                }
                else if ((keyword == tokNOTE) || (keyword == tokGET) || (keyword == tokINPUTHASH))
                {
                    parseCommaList("#R,L");
                }
                else if ((keyword == tokPLOT) ||
                         (keyword == tokSPRITEGEL) ||
                         (keyword == tokGELLOAD))
                {
                    parseCommaList("R,R");
                }
                else if ((keyword == tokDRAWSTRING) ||
                         (keyword == tokSETCOLOR) ||
                         (keyword == tokDRAWGEL) ||
                         (keyword == tokSPRITEMOVE))
                {
                    parseCommaList("R,R,R");
                }
                else if ((keyword == tokDRAWLINE) || (keyword == tokFILLRECT) || (keyword == tokDRAWRECT))
                {
                    parseCommaList("R,R,R,R");
                }
                else if ((keyword == tokGELGRAB))
                {
                    parseCommaList("R,R,R,R,R");
                }
                else if ((keyword == tokFILLROUNDRECT) || (keyword == tokDRAWROUNDRECT) ||
                         (keyword == tokFILLARC) || (keyword == tokDRAWARC) ||
                         (keyword == tokBLIT))
                {
                    parseCommaList("R,R,R,R,R,R");
                }
                else if ((keyword == tokREM) || (keyword == tokDATA))
                {
                    while ((lineOffset < lineLen) && (line.charAt(lineOffset) == ' '))
                        lineOffset++;
                    
                    token = line.substring(lineOffset);
                    if (token != null)
                        token = token.trim();
                    
                    //if (DEBUG)
                    //{
                    //    System.out.println("REM/DATA: line=" + line);
                    //    System.out.println("REM/DATA: token=" + token);
                    //    System.out.println("REM/DATA: token.length()=" + token.length());
                    //}
                    
                    lineOffset = lineLen;
                    
                    if ((token == null) || (token.length()==0))
                    {
                        if (keyword == tokREM)
                            token = "";
                        else
                            throw new BasicError(BasicError.SYNTAX_ERROR, "No data on line!");
                    }
                    
                    int len = token.length();
                    code[codeLen++] = (byte)(len & 0xff);
                    for (int i=0;i<len;i++)
                    {
                        code[codeLen++] = (byte)(token.charAt(i));
                    }
                }
                else if (keyword == tokIF)
                {
                    parseRValue(false);
                    token = GetToken();
                    
                    if (DEBUG)
                        System.out.println("token = " + token);
                    
                    if ((token != null) && (token.compareTo("THEN") == 0))
                    {
                        code[codeLen++] = (byte)tokTHEN;
                        statementRequired = true;
                        continue;   // We don't want to check for a Colon ":" before next statement
                    }
                    else
                    {
                        throw new BasicError(BasicError.SYNTAX_ERROR, "THEN expected");
                    }
                }
                else if ((keyword == tokPRINT) || (keyword == tokGOTO) || (keyword == tokGOSUB) || (keyword == tokSLEEP) || (keyword == tokTRAP) || (keyword == tokRESTORE) || (keyword == tokLOAD) || (keyword == tokENTER) || (keyword == tokSAVE) || (keyword == tokEDIT) || (keyword == tokDELETE))
                    parseRValue(false);
                else if (keyword == tokLIST)
                {
                    token = GetToken();
                    if (token != null)
                    {
                        PutToken(token);
                        parseRValue(false);
                        token = GetToken();
                        if (token != null)
                        {
                            if (token.compareTo(",") == 0)
                            {
                                code[codeLen++] = (byte)tokCOMMA;
                                parseRValue(false);
                            }
                            else
                            {
                                throw new BasicError(BasicError.COMMA_EXPECTED, "Comma expected");
                            }
                        }
                    }
                }
                else if (keyword == tokINPUT)
                {
                    parseRValue(false);
                    token = GetToken();
                    if ((token != null) && (token.compareTo(",") == 0))
                    {
                        code[codeLen++] = (byte)tokCOMMA;
                        parseLValue();
                    }
                    else
                        throw new BasicError(BasicError.COMMA_EXPECTED, "Comma expected");
                }
                else if (keyword == tokFOR)
                {
                    //PutToken(token);
                    parseLValue();
                    token = GetToken();
                    if ((token != null) && (token.compareTo("=") == 0))
                    {
                        code[codeLen++] = (byte)tokFOREQ;
                        parseRValue(false);
                        token = GetToken();
                        if ((token != null) && (token.compareTo("TO") == 0))
                        {
                            code[codeLen++] = (byte)tokTO;
                            parseRValue(false);
                            token = GetToken();
                            if (token != null)
                            {
                                if (token.compareTo("STEP") == 0)
                                {
                                    code[codeLen++] = (byte)tokSTEP;
                                    parseRValue(false);
                                }
                                else
                                {
                                    PutToken(token);
                                }
                            }
                        }
                        else
                        {
                            throw new BasicError(BasicError.SYNTAX_ERROR, "TO expected");
                        }
                    }
                    else
                    {
                        throw new BasicError(BasicError.SYNTAX_ERROR, "Assignment expected");
                    }
                }
            }
            else
            {
                if (DEBUG)
                    System.out.println("Variable: " + token);
                
                PutToken(token);                            // Put variable back - parseLValue() expects it!
                boolean arrayFlag = parseLValue();
                token = GetToken();
                if (token.compareTo("=") == 0)
                {
                    //code[codeLen++] = arrayFlag ? (byte)tokARRAY_WRITE : (byte)tokASSIGN;
                    code[codeLen++] = (byte)tokASSIGN;
                    parseRValue(false);
                }
                else
                {
                    throw new BasicError(BasicError.SYNTAX_ERROR, "Assignment expected");
                }
            }
            
            token = GetToken();
            if (token != null)
            {
                if (token.compareTo(":") == 0)
                {
                    statementRequired = true;
                    code[codeLen++] = (byte)tokCOLON;
                }
                else
                    throw new BasicError(BasicError.SYNTAX_ERROR, "Colon expected");
            }
        }

        if (statementRequired)
            throw new BasicError(BasicError.SYNTAX_ERROR, "Statement expected");
    }
    
  public static boolean parseLine(String inputLine, boolean correctPrompt)
  {
    if (DEBUG)
      System.out.println("BASIC.parseLine() entry");

    boolean okFlag;

    line = inputLine;
    lineLen = inputLine.length();
    lineOffset = 0;
    
    /*
     * When an error is detected the token is pushed back into the token queue (via nextToken)
     * this allows another section to continue checking with the token that caused the error
     *
     * e.g. IF X=1 THEN (THEN is an error during an expression ... terminate expression, push
     *      THEN back into queue and resume with processing the IF statement ... which can
     *      now get THEN as the next token.
     *
     * This means that nextToken buffer must be clear when parsing a new line since any value
     * contained within nextToken is a token from the previous line.
     */
    
    nextToken = null;               // Clear token buffer
    
    try
    {
      String token = GetToken();
      int lno;

      try
      {
        lno = Integer.parseInt(token);
      }
      catch (NumberFormatException e)
      {
        PutToken(token);
        lno = -1;
      }

      codeLen = 3;

      parseStatement();

      String t = GetToken();
      if (t != null)
          throw new BasicError(BasicError.SYNTAX_ERROR, "Trailing Junk: " + t);
      
      if (codeLen > 3)
      {
          code[codeLen++] = (byte)tokEOS;
          code[2] = (byte)(codeLen & 0xff);

          if (lno != -1)
              InsertLine(lno, codeLen, code);
          else
              RunProgram(code, codeLen);
      }
      else if (lno != -1)
      {
          RemoveLine(lno);
      }

      okFlag = true;
    }
    catch (BasicError e)
    {
      support.Error("Error " + e.errorNumber + ": " + e.getMessage() + " near " + line.substring(lineOffset-1));
 
      if (DEBUG)
      {
        System.out.println("parseLine> Error: " + line);
        System.out.print("                  ");
        for (int i=1;i<lineOffset;i++)
          System.out.print(" ");
        System.out.println("^");
        e.printStackTrace();
      }

      if ((e.errorNumber != BasicError.OUT_OF_MEMORY) && correctPrompt)
      {
        inputLine = support.GetLine("Correct>", line);
        okFlag = parseLine(inputLine, correctPrompt);
      }
      else
      {
        okFlag = false;
      }
    }
    catch (RuntimeException e)
    {
      support.Error("Error: " + e.getMessage() + " near " + line.substring(lineOffset-1));
        
      if (DEBUG)
      {
        System.out.println("parseLine> Error: " + line);
        System.out.print("                  ");
        for (int i=1;i<lineOffset;i++)
          System.out.print(" ");
        System.out.println("^");
        e.printStackTrace();
      }

      okFlag = false;
    }

    if (DEBUG)
      System.out.println("BASIC.parseLine() exit");

    return okFlag;
  }

    public static void List(DataOutput dataOutput, int lno1, int lno2, boolean editFlag)
    {
        //int tmpPC = progPC;

        int progPC = 0;

      listLoop:
        while (progPC < sourceLen)
        {
            int linePC = progPC;
            int lineNum = ((sourceProg[progPC++] & 0xff) << 8) +
                           (sourceProg[progPC++] & 0xff);
            int lineLen = (sourceProg[progPC++] & 0xff);
            int nextPC = linePC + lineLen;

            if (DEBUG)
                System.out.println("List: linePC=" + linePC + ", lineNum=" + lineNum + ", lineLen=" + lineLen);
            
            if ((lineNum >= lno1) && (lineNum <= lno2))
            {
                StringBuffer sb = new StringBuffer();
                
                sb.append(Integer.toString(lineNum));
                //sb.append(' ');

                boolean suppressColon = false;
                boolean spaceRequired = true;
                
                while (progPC < nextPC)
                {
                  int token = (sourceProg[progPC++] & 0xff);

                  if (DEBUG)
                      System.out.println("Token: " + token);

                  if (token < tokenTable.length)
                      if (((int)tokenTable[token].charAt(0) & 0x0080) == 0x0080)
                          spaceRequired = true;
                  
                  if (spaceRequired)
                  {
                    sb.append(' ');
                    spaceRequired = false;
                  }

                  if (token < tokenTable.length)
                      if (((int)tokenTable[token].charAt(0) & 0x0040) == 0x0040)
                          spaceRequired = true;

                  if (token == tokUBYTE)
                  {
                      int ival = (sourceProg[progPC++] & 0xff);
                      sb.append(ival);
                  }
                  else if (token == tokBYTE)
                  {
                      int ival = sourceProg[progPC++];
                      sb.append(ival);
                  }
                  else if (token == tokUWORD)
                  {
                      int ival = ((sourceProg[progPC++] & 0xff) << 8) + (sourceProg[progPC++] & 0xff);
                      sb.append(ival);
                  }
                  else if (token == tokINTEGER)
                  {
                    int ival = ((sourceProg[progPC++] & 0xff) << 24) +
                               ((sourceProg[progPC++] & 0xff) << 16) +
                               ((sourceProg[progPC++] & 0xff) << 8) +
                                (sourceProg[progPC++] & 0xff);
                    sb.append(ival);
                  }
                  else if (token == tokFLOAT)
                  {
                    int ival = ((sourceProg[progPC++] & 0xff) << 24) +
                               ((sourceProg[progPC++] & 0xff) << 16) +
                               ((sourceProg[progPC++] & 0xff) << 8) +
                                (sourceProg[progPC++] & 0xff);
                    sb.append(Float.toString(ival));
                  }
                  else if (token == tokVARIABLE)
                  {
                    int ival = sourceProg[progPC++] & 0xff;
                    if (DEBUG)
                        System.out.println("tokVARIABLE: " + ival);
                    sb.append(varName[ival]);
                  }
                  else if (token == tokSTRING)
                  {
                    sb.append("\"");
                    int len = sourceProg[progPC++] & 0xff;
                    for (int i=0;i<len;i++)
                      sb.append((char)(sourceProg[progPC++] & 0xff));
                    sb.append("\"");
                  }
                  else if ((token == tokREM) || (token == tokDATA))
                  {
                    sb.append(tokenTable[token].substring(3));
                    sb.append(' ');
                    int len = sourceProg[progPC++] & 0xff;
                    for (int i=0;i<len;i++)
                      sb.append((char)(sourceProg[progPC++] & 0xff));
                    //while (progPC < nextPC)
                    //    sb.append((char)(sourceProg[progPC++] & 0xff));
                  }
                  else if (token == tokASSIGN) // || (token == tokARRAY_WRITE))
                  {
                    sb.append("=");
                  }
                  //else if (token == tokCOLON)
                  //{
                  //    sb.append(":");
                  //}
                  else
                  {
                    //if (token == tokTHEN)
                    //  sb.append(' ');
                    
                    if (token != tokEOS)
                    {
                        if (token < tokenTable.length)
                            sb.append(tokenTable[token].substring(3));
                    }
                    
                    //if (token < tokenTable.length)
                    //    spaceRequired = ((int)tokenTable[token].charAt(0) & 0x0080) == 0x0080 ? true : false;
                    //else
                    //    System.out.println("Token: " + token + " is not inside keywordTable");
                  }
                }
                
                if (!editFlag)
                    sb.append('\n');
                
                String string = sb.toString();

                if (editFlag)
                {
                    String inputLine = support.GetLine("Edit>", string);
                    parseLine(inputLine, true);
                    break listLoop; // Cannot continue with List Loop - source has change so this loops variables are now inconsistent
                }
                else
                {
                    if (dataOutput != null)
                    {
                        try
                        {
                            for (int i=0;i<string.length();i++)
                                dataOutput.writeByte((byte)string.charAt(i));
                        }
                        catch (IOException e)
                        {
                            throw new BasicError(BasicError.IO_ERROR, "I/O Error");
                        }
                    }
                    else
                    {
                        if (DEVELOPMENT_MODE)
                            System.out.println(string);
                        support.PrintString(string);
                    }
                }

                if (progPC != nextPC)
                  throw new BasicError(BasicError.INTERNAL_ERROR, "List: Internal Error Line " + lineNum);
            }

            progPC = nextPC;
        }

      //progPC = tmpPC;
    }

  static private boolean stopProgramFlag;

  public static void StopProgram()
  {
    stopProgramFlag = true;
  }

  private static byte[] exeProg;
  private static int exeLen;
  private static int exePC;
  private static int exeNextLinePC;
  private static int trapPC;        // Execution point if error occurs
  private static int lastError ;
  private static int dataPC;        // Pointer for Read Data
  private static int dataOffset;    // Offset into current data statement
  private static int dataLen;       // Length of data in current statement
  
  static void RunProgram (byte[] code, int codeLen)
  {
    if (DEBUG)
    {
      System.out.println("BASIC.RunProgram() entry");

      System.out.println("code=" + code);
      System.out.println("codeLen=" + codeLen);
      System.out.println("sourceProg=" + sourceProg);
      System.out.println("sourceLen=" + sourceLen);
    }

    int lineNum = 0;

    degFlag = false;

    exeProg = code;
    exeLen = codeLen;
    exePC = 0;

    trapPC = -1;
    lastError = 0;
    dataPC = 0;
    dataOffset = -1;
    dataLen = -1;
                    
    valueSP = -1;       // Clear operand stack
    operatorSP = -1;    // Clear operator stack
    controlSP = -1;     // Clear control stack

    dirEnum = null;
 
    stopProgramFlag = false;

    if (DEVELOPMENT_MODE)
      System.out.println("Program loop started");
              
      programLoop:
        while (exePC < exeLen)
        {
            //System.out.println("controlSP=" + controlSP + ", valueSP=" + valueSP + ", operatorSP=" + operatorSP);
            
            try
            {
                int lineLen;

                lineNum = ((exeProg[exePC++] & 0xff) << 8) + (exeProg[exePC++] & 0xff);
                lineLen = (exeProg[exePC++] & 0xff);
                exeNextLinePC = exePC + lineLen - 3;

                if (DEBUG)
                {
                    System.out.println("++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++");
                    System.out.println("Line " + lineNum);
                }
          
            lineLoop:
                while (exePC < exeNextLinePC)
                {
                    if (stopProgramFlag)
                        break programLoop;
        
                    int opcode = (exeProg[exePC++] & 0xff);

                    if (DEBUG)
                    {
                        System.out.println("------------------------------------------------------------");
                        System.out.println("Opcode = " + opcode);
                        System.out.println("OperatorStack: " + DumpOperatorStack());
                        System.out.println("OperandStack: " + DumpOperandStack());
                    }
                    
                        switch (opcode)
                        {
                            case tokBYTE :
                            {
                                int ival = exeProg[exePC++];
                                PushOperand(ival, null, TYPE_INTEGER, CLASS_CONSTANT);
                                break;
                            }

                            case tokUBYTE :
                            {
                                int ival = (exeProg[exePC++] & 0xff);
                                PushOperand(ival, null, TYPE_INTEGER, CLASS_CONSTANT);
                                break;
                            }

                            case tokUWORD :
                            {
                                int ival = ((exeProg[exePC++] & 0xff) << 8) +
                                            (exeProg[exePC++] & 0xff);
                                PushOperand(ival, null, TYPE_INTEGER, CLASS_CONSTANT);
                                break;
                            }

                            case tokINTEGER :
                            {
                                int ival = ((exeProg[exePC++] & 0xff) << 24) +
                                           ((exeProg[exePC++] & 0xff) << 16) +
                                           ((exeProg[exePC++] & 0xff) << 8) +
                                            (exeProg[exePC++] & 0xff);
                                PushOperand(ival, null, TYPE_INTEGER, CLASS_CONSTANT);
                                break;
                            }

                            case tokFLOAT :
                            {
                                int ival = ((exeProg[exePC++] & 0xff) << 24) +
                                    ((exeProg[exePC++] & 0xff) << 16) +
                                    ((exeProg[exePC++] & 0xff) << 8) +
                                    (exeProg[exePC++] & 0xff);
                                PushOperand(ival, null, TYPE_FLOAT, CLASS_CONSTANT);
                                break;
                            }

                            case tokVARIABLE :
                            {
                                int ival = exeProg[exePC++] & 0xff;

                                if (DEBUG)
                                    System.out.println("PUSH_VARIABLE: " + varName[ival] + " (" + ival + ")");

                                PushOperand(ival, null, varType[ival], CLASS_VARIABLE);
                                break;
                            }

                            case tokSTRING :
                            {
                                StringBuffer sb = new StringBuffer();

                                int len = exeProg[exePC++] & 0xff;
                                for (int i=0;i<len;i++)
                                    sb.append((char)(exeProg[exePC++] & 0xff));

                                if (DEBUG)
                                    System.out.println("PUSH_STRING: " + sb.toString());
                  
                                PushOperand(0, sb.toString(), TYPE_STRING, CLASS_CONSTANT);
                                break;
                            }

                            case tokREM :
                            case tokDATA :
                            {
                                exePC = exeNextLinePC;
                                break;
                            }
                            
                            case tokASSIGN :
                                AddOperator(opcode, 1, 1);
                                break;
                    
                            case tokMAKEREF :
                                AddOperator(opcode, 15, 2);
                                break;
                    
                            //case tokCOLON :
                            case tokEOS :
                                AddOperator(opcode, 0, 0);
                                commaCount = 0; // Comma Count won't be reset for commaLists outside of brackets (e.g. Keywords)
                                break;
                    
                            default :
                                int prior = tokenTable[opcode].charAt(0);
                                int onPrior = (prior & 0xf000) >> 12;
                                int offPrior = (prior & 0x0f00) >> 8;
                                AddOperator(opcode, onPrior, offPrior);
                                break;
                        }
                }

                if (exePC != exeNextLinePC)
                {
                    throw new BasicError(BasicError.INTERNAL_ERROR, "Internal Error: exePC <> exeNextLinePC");
                }

                Thread.yield();
            }
            catch (BasicError e)
            {
                if (DEBUG)
                {
                    System.out.print("ERROR> ");
                    DumpOperatorStack();
                    DumpOperandStack();
                }
                
                if (trapPC != -1)
                {
                    valueSP = -1;       // Must clear operand stack
                    operatorSP = -1;    // Must clear operator stack
                    // controlSP = -1;
                    lastError = e.errorNumber;
                    exePC = trapPC;
                    trapPC = -1;
                    continue programLoop;       // Start new line
                }

                support.Error("Error: " + e.getMessage() + " at line " + lineNum);

                if (DEVELOPMENT_MODE)
                {
                    System.out.println("RunProgram> Error: " + e.getMessage() + " at line " + lineNum + "\n");
                    DumpOperatorStack();
                    DumpOperandStack();
                    e.printStackTrace();
                }

                break programLoop;
            }
            catch (FloatException e)
            {
                if (trapPC != -1)
                {
                    valueSP = -1;       // Must clear operand stack
                    operatorSP = -1;    // Must clear operator stack
                    // controlSP = -1;
                    lastError = e.errorNumber;
                    exePC = trapPC;
                    trapPC = -1;
                    continue programLoop;       // Start new line
                }

                support.Error("Error: " + e.getMessage() + " at line " + lineNum);

                break programLoop;
            }
            catch (Exception e)
            {
                Class c = e.getClass();
                support.Error("Error: " + c.getName() + " at line " + lineNum);
                
                if (DEVELOPMENT_MODE)
                {
                    support.PrintString("RunProgram> Error: " + c.getName() + " at line " + lineNum + "\n");
                    DumpOperatorStack();
                    DumpOperandStack();
                    e.printStackTrace();
                }
                
                break programLoop;
            }
        }

    if (DEVELOPMENT_MODE)
      System.out.println("Program loop finished");

    support.CloseAllFiles();

    if (stopProgramFlag)
    {
      support.Message("Stopped at line " + lineNum);
    }

    if (DEBUG)
      System.out.println("BASIC.RunProgram() exit");
  }
}
