package com.mobilebasic;

/*
 * Mobile BASIC 1.9 Copyright (c) 2003-2010, David Firth
 *
 * This software is released as Open Source and you are free to modify it
 * and build derived versions for your own use. If you have made any changes
 * that you would like included in the main version then they should be
 * sent to the address below.
 *
 * Patches: E-mail to david@mobilebasic.com
 * Website: http://www.mobilebasic.com/
 */

import javax.microedition.midlet.*;
import javax.microedition.lcdui.*;

import java.io.*;
import java.util.Enumeration;
import java.util.Hashtable;

public class BasicCanvas extends Canvas implements Runnable {
    
    //private static final boolean DEBUG_FLAG = false;
    
    public static final int GAME_UP = 0x0001;
    public static final int GAME_DOWN = 0x0002;
    public static final int GAME_LEFT = 0x0004;
    public static final int GAME_RIGHT = 0x0008;
    public static final int GAME_FIRE = 0x0010;
    public static final int GAME_A = 0x0020;
    public static final int GAME_B = 0x0040;
    public static final int GAME_C = 0x0080;
    public static final int GAME_D = 0x0100;

    public int gameActionBits = 0x0000;
    
    private int widthInPixels;
    private int heightInPixels;

    public Font font;
    private int lineHeight;
    private int charWidth;
    private int widthInChars;
    private int heightInChars;
    private int nlines;

    private Image graphicsImage;
    private Image offScreenImage;
    public Graphics graphicsGc;
    private Graphics offScreenGc;

    /*
     * Sprite / Gobs (Graphic Objects) / Gels (Graphic Elements)
     *
     * Updating the sprite is very costly since there is no
     * direct access to the image data. Consequently all
     * images should be drawn first. If we have a image
     * per sprite then there could be a lot of duplication
     * particularily if there are many similar sprites.
     */

    private Hashtable gelHashtable = new Hashtable();
    private Hashtable spriteHashtable = new Hashtable();

    private Thread thread;
  
    BasicCanvas()
    {
        super();

        widthInPixels = getWidth();
        heightInPixels = getHeight();

        graphicsImage = Image.createImage(widthInPixels, heightInPixels);
        graphicsGc = graphicsImage.getGraphics();
        offScreenImage = Image.createImage(widthInPixels, heightInPixels);
        offScreenGc = offScreenImage.getGraphics();

        SetFontSize(0);
    }

    void SetFontSize(int fontSize)
    {
      /*
       * FACE_ {MONOSPACE|PROPORTIONAL|SYSTEM}
       * SIZE_ {SMALL|MEDIUM|LARGE}
       * STYLE_ {BOLD|ITALIC|PLAIN|UNDERLINE}
       */

      //System.out.println("fontSize = " + fontSize);

      switch (fontSize)
      {
        case 1 :
          //System.out.println("SMALL Font");
          font = Font.getFont(Font.FACE_MONOSPACE,
                              Font.STYLE_PLAIN,
                              Font.SIZE_SMALL);
          break;
        case 2 : default :
          //System.out.println("MEDIUM Font");
          font = Font.getFont(Font.FACE_MONOSPACE,
                              Font.STYLE_PLAIN,
                              Font.SIZE_MEDIUM);
          break;
        case 3 :
          //System.out.println("LARGE Font");
          font = Font.getFont(Font.FACE_MONOSPACE,
                              Font.STYLE_PLAIN,
                              Font.SIZE_LARGE);
          break;
      }

      charWidth = font.charWidth('W');
      widthInChars = widthInPixels / charWidth;

      if ((fontSize == 0) && (widthInChars < 12))
      {
        //System.out.println("Auto: Changing to SMALL Font");
        font = Font.getFont(Font.FACE_MONOSPACE,
                            Font.STYLE_PLAIN,
                            Font.SIZE_SMALL);
        charWidth = font.charWidth('W');
        widthInChars = widthInPixels / charWidth;
      }
        
      lineHeight = font.getHeight();

      heightInPixels = getHeight();
      heightInChars = heightInPixels / lineHeight;

      graphicsGc.setFont(font);
      offScreenGc.setFont(font);
    }

    public void Init()
    {
      gelHashtable.clear();
      spriteHashtable.clear();
    }

    /*
     * Called when the Canvas is getting the Focus again.
     * If pressing FIRE causes the BASIC program to display a form then the canvas
     * never gets key release events and the gameActionBits will still indicate
     * that the Fire button is pressed. This routine allows us to reset the
     * status before it is re-focussed.
     */
    public void Focus()
    {
        gameActionBits = 0x0000;
    }

    int yposText = 0;
    int xposText = 0;

    public void printString(String string, int firstChar, int cursorChar, boolean wrapEnabled, boolean pauseEnabled)
    {
        heightInPixels = getHeight();
        heightInChars = heightInPixels / lineHeight;

        int strlen = string.length();

        for (int i=firstChar;i<strlen;i++)
        {
            char ch = string.charAt(i);

            /*
             * Start newline if newline character or screen wrap
             */

            if ((ch == '\n') || ((xposText + charWidth) > widthInPixels))
            {
              if (wrapEnabled)
              {
                xposText = 0;
                yposText += lineHeight;

                if ((yposText + lineHeight) > heightInPixels)
                {
                  if (pauseEnabled)
                  {
                    if (nlines >= heightInChars)
                    {
                      while (keyPressed == 0) // Wait for key press
                        Thread.yield();

                      while (keyPressed != 0) // Wait for key release
                        Thread.yield();

                      nlines = 0;
                    }
                  }
                  else
                  {
                    nlines = 0;
                  }

                  Blit(0, lineHeight,
                       widthInPixels, heightInPixels-lineHeight,
                       0, 0);
                  yposText -= lineHeight;
                  nlines++;
                }
              }
              else
                break;
            }

            /*
             * Clear line if at start of line
             */

            if (xposText == 0)
            {
              graphicsGc.setColor(0xffffff);
              graphicsGc.fillRect(0, yposText, widthInPixels, lineHeight);
              graphicsGc.setColor(0x000000);
            }

            /*
             * Draw Character if not newline
             */

            if (ch != '\n')
            {
                if (i == cursorChar)
                {
                    graphicsGc.setColor(0x000000);
                    graphicsGc.fillRect(xposText, yposText,
                                        charWidth, lineHeight);
                    graphicsGc.setColor(0xffffff);
                }
                else
                {
                    graphicsGc.setColor(0x000000);
                }

                graphicsGc.drawChar(ch,
                                    xposText + charWidth/2,
                                    yposText,
                                    Graphics.TOP | Graphics.HCENTER);
                xposText += charWidth;
            }
        }

        //xposText = widthInPixels; // Start new line
    }
    
    private static char lowerCaseKeys[][] =
    {
        { ' ', '\n', 0x0001, 0x0002, 0x0008, 0x0009 },          // 0
        { '%', '$', ',', '\"', '.', ':', '?', '\'', '(', ')', '!', '@', ';', '{', '}', '[', ']' },  // 1
        { 'a', 'b', 'c' },                                      // 2
        { 'd', 'e', 'f' },                                      // 3
        { 'g', 'h', 'i' },                                      // 4
        { 'j', 'k', 'l' },                                      // 5
        { 'm', 'n', 'o' },                                      // 6
        { 'p', 'q', 'r', 's' },                                 // 7
        { 't', 'u', 'v' },                                      // 8
        { 'w', 'x', 'y', 'z' },                                 // 9
        { '/', '+', '-', '%', '=', '<', '>', '&', '|', '^' },   // STAR
        { 0x0003, 0x0004 }                                      // POUND
    };

    private static char upperCaseKeys[][] =
    {
        { ' ', '\n', 0x0001, 0x0002, 0x0008, 0x0009 },          // 0
        { '%', '$', ',', '\"', '.', ':', '?', '\'', '(', ')', '!', '@', ';', '{', '}', '[', ']' },  // 1
        { 'A', 'B', 'C' },                                      // 2
        { 'D', 'E', 'F' },                                      // 3
        { 'G', 'H', 'I' },                                      // 4
        { 'J', 'K', 'L' },                                      // 5
        { 'M', 'N', 'O' },                                      // 6
        { 'P', 'Q', 'R', 'S' },                                 // 7
        { 'T', 'U', 'V' },                                      // 8
        { 'W', 'X', 'Y', 'Z' },                                 // 9
        { '/', '+', '-', '=', '<', '>', '&', '|', '^' },        // STAR
        { 0x0003, 0x0004 }                                      // POUND
    };

    private static char keys[][] = lowerCaseKeys;
    
    private StringBuffer inputLine = new StringBuffer(128);
    private String line = null;
    private int minCursorPos = 0;
    private int windowPos = 0;
    private int cursorPos = 0;
    
    private boolean inputEnabled = false;

    public String GetLine(String prompt, String defaultText)
    {
        line = null;
        inputLine.delete(0, inputLine.length());
        inputLine.append(prompt);
        minCursorPos = prompt.length();
        
        if (defaultText != null)
            inputLine.append(defaultText);
//        cursorPos = inputLine.length() - 1;
        
        inputEnabled = true;
        
        keyTyped(-1);    // Make sure line is cleared and cursor is visible

        while (true)
        {
            String t = line;
            
            if (t != null)
            {
                inputEnabled = false;
                line = null;
                return t;
            }
            
            Thread.yield();
        }
    }
    
    public void keyTyped(int keyCode)
    {
        //int xpos = 0;
        //int ypos = heightInChars-1;

        if (keyCode == -1)
        {
//            inputLine.delete(0, inputLine.length());
            windowPos = 0;
            cursorPos = minCursorPos;
            //cursorPos = inputLine.length();

          /*
           * The line that we will be editing will be displayed without
           * wrapping. If the cursor is not already at the start of the
           * line then we need to print a carriage return character with
           * wrapping enabled.
           * Note: surely this is always going to be true
           *
           * printString sets xposText to end of line which will cause
           * a newline to be issued immediately if wrapping is enabled.
           * This text entry disables wrapping so we need to set
           * xposText to zero manually.
           */

          if (xposText > 0)
          {
            printString("\n", 0, -1, true, false); // With wrapping enabled
            xposText = 0;
          }
        }
        else if (keyCode == '\n')
        {
            String s = inputLine.toString();
            
            inputLine.delete(0, inputLine.length());

            printString(s + "\n", 0, -1, true, false);
            line = s.substring(minCursorPos);
//            if (s.length() > 0)
//                basicCanvasListener.parseLine(s);
            
            windowPos = minCursorPos;
            cursorPos = minCursorPos;
        }
        else if (keyCode == 0x0001)
        {
            if (cursorPos > minCursorPos)
                cursorPos--;
        }
        else if (keyCode == 0x0002)
        {
            cursorPos++;
            if (cursorPos > inputLine.length())
                cursorPos = inputLine.length();
        }
        else if (keyCode == 0x0008)                 // Backspace
        {
            /*
            inputLine.deleteCharAt(cursorPos);
            if (cursorPos >= inputLine.length())
                cursorPos = inputLine.length() - 1;
             */
            if (cursorPos > minCursorPos)
            {
                inputLine.deleteCharAt(cursorPos-1);
                cursorPos = cursorPos - 1;
            }
        }
        else if (keyCode == 0x0009)                 // Delete
        {
            if (cursorPos < inputLine.length())
            {
                inputLine.deleteCharAt(cursorPos);
                if (cursorPos > inputLine.length())
                    cursorPos = inputLine.length();
//                if (cursorPos < 0)
//                    cursorPos = 0;
            }
        }
        else
        {
            inputLine.insert(cursorPos++, (char)keyCode);
        }

        /*
         * Display Updated Line
         */

        if (cursorPos < windowPos)
        {
            windowPos = cursorPos;
        }
        else if (cursorPos >= (windowPos + widthInChars))
        {
            windowPos = cursorPos - widthInChars + 1;
        }

        printString(inputLine.toString() + " ", windowPos, cursorPos, false, false);
        xposText = 0;
    }
    
    private int keyPressed = 0;
    private int lastKeyIndex = -1;
    private int keyClicks = 0;
    private long keyTime = 0;
    private long specialTime = 0;

    private void alphaPressed(int keyCode)
    {
        synchronized (this)
        {
            int keyIndex = -1;

            switch (keyCode)
            {
                case Canvas.KEY_NUM0 :
                    keyIndex = 0;
                    break;
                case Canvas.KEY_NUM1 :
                    keyIndex = 1;
                    break;
                case Canvas.KEY_NUM2 :
                    keyIndex = 2;
                    break;
                case Canvas.KEY_NUM3 :
                    keyIndex = 3;
                    break;
                case Canvas.KEY_NUM4 :
                    keyIndex = 4;
                    break;
                case Canvas.KEY_NUM5 :
                    keyIndex = 5;
                    break;
                case Canvas.KEY_NUM6 :
                    keyIndex = 6;
                    break;
                case Canvas.KEY_NUM7 :
                    keyIndex = 7;
                    break;
                case Canvas.KEY_NUM8 :
                    keyIndex = 8;
                    break;
                case Canvas.KEY_NUM9 :
                    keyIndex = 9;
                    break;
                case Canvas.KEY_STAR :
                    keyIndex = 10;
                    break;
                case Canvas.KEY_POUND :
                    keyIndex = 11;
                    break;
                default :
                    keyIndex = -1;
                    break;
            }
    
            if (keyIndex != -1)
            {
                keyTime = System.currentTimeMillis();
    
                if (keyIndex == lastKeyIndex)
                {
                    keyClicks++;
                    if (keyClicks == keys[keyIndex].length)
                        keyClicks = 0;
                }
                else if (lastKeyIndex != -1)
                {
                    keyCode = (int)keys[lastKeyIndex][keyClicks];

                    if (keyCode == 0x0003)      // Switch to lower case
                    {
                        keys = lowerCaseKeys;
                    }
                    else if (keyCode == 0x0004) // Switch to upper case
                    {
                        keys = upperCaseKeys;
                    }
                    else
                    {
                        keyTyped(keyCode);
                    }

//                  keyTime = 0;
                    keyClicks = 0;
                }
            }

            lastKeyIndex = keyIndex;
        }
    }
  
    private void specialPressed(int keyCode)
    {
        boolean longPress = (System.currentTimeMillis() - specialTime) > 500;
        int ch;
      
        synchronized (this)
        {
            if (lastKeyIndex != -1)
            {
                int keyCode2 = (int)keys[lastKeyIndex][keyClicks];

                keyTyped(keyCode2);
                lastKeyIndex = -1;
                keyClicks = 0;
            }
                    
            keyTime = 0;

            //System.out.println("keyCode=" + keyCode);
            switch (getGameAction(keyCode))
            {
                case Canvas.LEFT :
                case Canvas.UP :
                    keyTyped(longPress ? 0x0008 : 0x0001);
                    break;
                case Canvas.RIGHT :
                case Canvas.DOWN :
                    keyTyped(longPress ? 0x0009 : 0x0002);
                    break;
                case Canvas.FIRE :
                case Canvas.GAME_A :
                case Canvas.GAME_B :
                case Canvas.GAME_C :
                case Canvas.GAME_D :
                    keyTyped('\n');
                    break;
                default :
                    break;
            }
            //System.out.println("Done");
        }
    }

    private boolean terminateFlag;

    public void StartThread()
    {
        terminateFlag = false;
        thread = new Thread(this);
        thread.start();
    }
    
    public void StopThread()
    {
        terminateFlag = true;
        
        try
        {
            thread.join();
        }
        catch (InterruptedException e)
        {
        }
        
        thread = null;
    }
    
    public void run()
    {
        //System.out.println("BasicCanvas thread starting");

        while (!terminateFlag)
        {
            try
            {
                if (keyTime > 0)
                {
                    synchronized (this)
                    {
                        long currentTime = System.currentTimeMillis();

                        /*
                         * Handle Long Press First
                         */
                
                        //if ((currentTime > (keyTime + 250)) && (keyPressed >= 0x30 && keyPressed <= 0x39))
                        if ((currentTime > (keyTime + 500)) && (keyPressed > 0))
                        {
                            keyTime = 0;
                            keyClicks = 0;
                            lastKeyIndex = -1;
                            keyTyped(keyPressed);
                        }
                        else if (currentTime > (keyTime + 1000))
                        {
                            int keyCode = (int)keys[lastKeyIndex][keyClicks];
                            keyTime = 0;
                            keyClicks = 0;
                            lastKeyIndex = -1;

                            if (keyCode == 0x0003)      // Switch to lower case
                            {
                                keys = lowerCaseKeys;
                            }
                            else if (keyCode == 0x0004) // Switch to upper case
                            {
                                keys = upperCaseKeys;
                            }
                            else
                            {
                                keyTyped(keyCode);
                            }
                        }
                    }
                }

                repaint();
          
                Thread.sleep(1000/10); // i.e. Refresh 10 times per second
            }
            catch (InterruptedException e)
            {
            }
            catch (ArrayIndexOutOfBoundsException e)
            {
                //if (DEBUG_FLAG)
                //{
                //    System.out.println("keys[" + lastKeyIndex + "][" + keyClicks + "]");
                //    System.out.println(e.getMessage());
                //    e.printStackTrace();
                //}
            }
            catch (Exception e)
            {
                //if (DEBUG_FLAG)
                //{
                //    System.out.println(e.getMessage());
                //    e.printStackTrace();
                //}
            }
        }
      
        //System.out.println("BasicCanvas Thread Finished");
    }
  
    protected void keyPressed(int keyCode)
    {
        keyPressed = keyCode;
        
        /*
         * Always decode game actions
         */
   
        switch (getGameAction(keyCode))
        {
            case Canvas.UP :
                gameActionBits = GAME_UP;
                break;
            case Canvas.DOWN :
                gameActionBits = GAME_DOWN;
                break;
            case Canvas.LEFT :
                gameActionBits = GAME_LEFT;
                break;
            case Canvas.RIGHT :
                gameActionBits = GAME_RIGHT;
                break;
            case Canvas.FIRE :
                gameActionBits = GAME_FIRE;
                break;
            case Canvas.GAME_A :
                gameActionBits = GAME_A;
                break;
            case Canvas.GAME_B :
                gameActionBits = GAME_B;
                break;
            case Canvas.GAME_C :
                gameActionBits = GAME_C;
                break;
            case Canvas.GAME_D :
                gameActionBits = GAME_D;
                break;
            default :
                break;
        }

        if (inputEnabled)
        {
            if (keyCode > 0)
                alphaPressed(keyCode);
            else
                specialTime = System.currentTimeMillis();
        }
    }

    protected void keyReleased(int keyCode)
    {
        if (inputEnabled)
        {
            if (keyCode < 0)
            {
                specialPressed(keyCode);
            }
        }
        
        keyPressed = 0;
        gameActionBits = 0x0000;
    }

    /*
    protected void pointerPressed(int x, int y)
    {
    }

    protected void pointerReleased(int x, int y)
    {
    }

    protected void pointerDragged(int x, int y)
    {
    }
     */

    public void Blit(int x, int y, int w, int h, int newX, int newY)
    {
        int clipX = graphicsGc.getClipX();
        int clipY = graphicsGc.getClipY();
        int clipWidth = graphicsGc.getClipWidth();
        int clipHeight = graphicsGc.getClipHeight();
        
        offScreenGc.drawImage(graphicsImage, -x, -y, Graphics.TOP | Graphics.LEFT);
        graphicsGc.setClip(newX, newY, w, h);
        graphicsGc.drawImage(offScreenImage, newX, newY, Graphics.TOP | Graphics.LEFT);
        graphicsGc.setClip(clipX, clipY, clipWidth, clipHeight);
    }

    public void GelLoad(String gelName, String resourceName)
    {
      try
      {
        Image image = Image.createImage("/" + resourceName);
        gelHashtable.put(gelName, image);
      }
      catch (IOException e)
      {
      }
    }

    public void GelGrab(String gelName, int x, int y, int w, int h)
    {
        Image image = Image.createImage(w, h);
        Graphics gc = image.getGraphics();
        gc.drawImage(graphicsImage, -x, -y, Graphics.TOP | Graphics.LEFT);
        gelHashtable.put(gelName, image);
    }

    public int GelWidth(String gelName)
    {
      int width;

      Image image = (Image)gelHashtable.get(gelName);
      if (image != null)
        width = image.getWidth();
      else
        width = 0;

      return width;
    }

    public int GelHeight(String gelName)
    {
      int height;

      Image image = (Image)gelHashtable.get(gelName);
      if (image != null)
        height = image.getHeight();
      else
        height = 0;

      return height;
    }

    public void DrawGel(String gelName, int x, int y)
    {
        Image image = (Image)gelHashtable.get(gelName);
        if (image != null)
          graphicsGc.drawImage(image, x, y, Graphics.TOP | Graphics.LEFT);
    }

    public void SpriteGEL(String spriteName, String gelName)
    {
        Image image = (Image)gelHashtable.get(gelName);
        if (image != null)
        {
          Object[] obj = (Object[])spriteHashtable.get(spriteName);
          if (obj != null)
          {
            obj[0] = image;
          }
          else
          {
            int[] pos = new int[2];
            pos[0] = 0;	// xpos
            pos[1] = 0;	// ypos

            obj = new Object[2];
            obj[0] = image;
            obj[1] = pos;

            spriteHashtable.put(spriteName, obj);
          }
        }
        else
        {
          throw new BasicError(BasicError.INVALID_GEL, "Invalid GEL");
        }
    }

    public void SpriteMove(String spriteName, int xpos, int ypos)
    {
      Object[] obj = (Object[])spriteHashtable.get(spriteName);
      if (obj != null)
      {
        int[] pos = (int[])obj[1];
        pos[0] = xpos;
        pos[1] = ypos;
      }
      else
      {
        throw new BasicError(BasicError.INVALID_SPRITE, "Invalid Sprite");
      }
    }

    public int SpriteHit(String spriteName1, String spriteName2)
    {
      int hitFlag = 0;

      Object[] sprite1 = (Object[])spriteHashtable.get(spriteName1);
      if (sprite1 != null)
      {
        Object[] sprite2 = (Object[])spriteHashtable.get(spriteName2);
        if (sprite2 != null)
        {
          Image image1 = (Image)sprite1[0];
          Image image2 = (Image)sprite2[0];
          int[] pos1 = (int[])sprite1[1];
          int[] pos2 = (int[])sprite2[1];

          int x1 = pos1[0];
          int y1 = pos1[1];
          int w1 = image1.getWidth();
          int h1 = image1.getHeight();

          int x2 = pos2[0];
          int y2 = pos2[1];
          int w2 = image2.getWidth();
          int h2 = image2.getHeight();

          if (((x2 > x1) && (x2 < (x1 + w1))) ||
              ((x1 > x2) && (x1 < (x2 + w2))))
          {
            if (((y2 > y1) && (y2 < (y1 + h1))) ||
                ((y1 > y2) && (y1 < (y2 + h2))))
              hitFlag = 1;
          }
        }
        else
        {
          throw new BasicError(BasicError.INVALID_SPRITE, "Invalid Sprite");
        }
      }
      else
      {
        throw new BasicError(BasicError.INVALID_SPRITE, "Invalid Sprite");
      }

      return hitFlag;
    }
 
    private char[] chArray = new char[1];

    protected void paint(Graphics g)
    {
        /*
        try
        {
         */
        
        offScreenGc.drawImage(graphicsImage, 0, 0, Graphics.TOP | Graphics.LEFT);

        Enumeration spriteEnumeration = spriteHashtable.elements();
        while (spriteEnumeration.hasMoreElements())
        {
          Object[] obj = (Object[])spriteEnumeration.nextElement();
          Image image = (Image)obj[0];
          int[] pos = (int[])obj[1];
          offScreenGc.drawImage(image, pos[0], pos[1],
                                Graphics.TOP | Graphics.LEFT);
        }

        /*
         * Overlay Keyboard Character Entry
         */ 

        if (keyTime > 0)
        {
            synchronized (this)
            {
                int xpos = widthInPixels;
                int ypos = 0; // lineHeight / 2;
                int nkeys = keys[lastKeyIndex].length;
                int ikey = 0;
            
                while (ikey < nkeys)
                {
                    char key = keys[lastKeyIndex][ikey];

                    String s;
                
                    if (key == 0x0003)
                        s = "<abc>";
                    else if (key == 0x004)
                        s = "<ABC>";
                    else if (key == '\n')
                        s = "<cr>";
                    else if (key == 0x0008)
                        s = "<bs>";
                    else if (key == 0x0009)
                        s = "<del>";
                    else if (key == 0x0001)
                        s = "<-";
                    else if (key == 0x0002)
                        s = "->";
                    else if (key == ' ')
                        s = "<sp>";
                    else
                    {
                        chArray[0] = key;
                        s = new String(chArray);
                    }
                
                    int cellWidth = font.stringWidth(s);
                
                    if (xpos+cellWidth >= (widthInPixels-5))    // Leave at least 5 pixel border at right edge
                    {
                        xpos = 0;
                        ypos += lineHeight;
                    
                        offScreenGc.setColor(0xffffff);
                        offScreenGc.fillRect(xpos, ypos, widthInPixels-1, lineHeight+2);
                        offScreenGc.setColor(0x000000);
                        offScreenGc.drawRect(xpos, ypos, widthInPixels-1, lineHeight+2);
                        xpos += 5;                    // Leave 5 pixel border at left edge
                    }
            
                    int cellLeft = xpos+1;
                    int cellCenter = cellLeft + cellWidth/2;
                
                    if (ikey == keyClicks)
                    {
                        offScreenGc.setColor(0x000000);
                        offScreenGc.fillRect(cellLeft, ypos+1, cellWidth, lineHeight);
                        offScreenGc.setColor(0xffffff);
                    }
                    else
                    {
                        offScreenGc.setColor(0x000000);
                    }
                
                    offScreenGc.drawString(s, cellCenter, ypos+1,
                                           Graphics.TOP | Graphics.HCENTER);
                    xpos += cellWidth;
                    ikey++;
                }
            }
        }
        
        g.drawImage(offScreenImage, 0, 0, Graphics.TOP | Graphics.LEFT);
        /*
        }
        catch (Exception e)
        {
            System.out.println(e.getMessage());
            e.printStackTrace();
        }
         */
    }
}
