package com.mobilebasic;

/*
 * Mobile BASIC 1.9 Copyright (c) 2003-2010, David Firth
 *
 * This software is released as Open Source and you are free to modify it
 * and build derived versions for your own use. If you have made any changes
 * that you would like included in the main version then they should be
 * sent to the address below.
 *
 * Patches: E-mail to david@mobilebasic.com
 * Website: http://www.mobilebasic.com/
 */

import javax.microedition.midlet.*;
import javax.microedition.lcdui.*;
import javax.microedition.rms.*;
import javax.microedition.io.*;

import java.io.*;
import java.util.Calendar;
import java.util.Date;
import java.util.Enumeration;
import java.util.Random;
import java.util.Vector;

/*
 * Fixed and Modifications Since Release 1.0
 *
 * 1. Separated BASIC Interpreter into new BASIC class. The intention is that it can now be
 *    used with an applet in a web page.
 * 2. Fixed Bug that prevented the Stop option on the menu from stopping the program.
 *    > 10 FOR X%=1 TO 10:X%=0:NEXT X%
 *    Basically the stop flag was only checked on the start of each new line. If the
 *    current line produced a loop in such a way that the line never completed then
 *    the stop flag was also never tested. The test has been moved to the start of each
 *    new statement.
 */

/*
 * Instruction
 *   Opcode
 *   Operand
 *     Integer Constant
 *     Float Constant
 *     String Constant
 *     Integer Variable
 *     Float Variable
 *     String Variable
 *     Line Number
 */

public class Main extends javax.microedition.midlet.MIDlet implements Runnable, CommandListener, BasicSupport
{
    private static final String BUILD_DATE = "2010/12/28";

    private static final boolean SAVE_COMMAND_ENABLED = false;
    private static final boolean HELP_LIST = true;

    private static Vector helpList;

    private Thread thread = null;
    private Runtime runtime = Runtime.getRuntime();

    private Alert alert;
    
    private Command exitCommand;
    private Command backCommand;
    private Command stopCommand;

    private Command showHelpListCommand;
    private Command loadCommand;
    private Command enterCommand;
    private Command deleteCommand;
    private Command sizeCommand;
    private Command saveCommand;
    private Command aboutCommand;
    private Command prefsCommand;
    
    /*
     * About Form
     */
    
    private Form aboutForm;

    /*
     * Preferences Form
     */

    private Form prefsForm;
    private ChoiceGroup fontSizeField;
    private ChoiceGroup textPauseField;

    BasicCanvas canvas;
    private int listFunc;
    private List list = null;
    private List exitList;
    private Graphics gc;
    
    Display display; 

    BASIC basic;
    
    Calendar calendar = Calendar.getInstance();

    boolean textPauseFlag = false;
    private boolean AutorunFlag;

    public void startApp()
    {
      canvas.StartThread();
      thread = new Thread(this);
      thread.start();
    }
    
    public void pauseApp()
    {
      canvas.StopThread();
      thread = null;
    }
    
    public void destroyApp(boolean unconditional)
    {
        canvas.StopThread();
        thread = null;
    }

    private void Init(boolean autorunFlag)
    {
        basic = new BASIC(this, 16384);

        /*
         * Create About Form
         */

        aboutForm = new Form(com.mobilebasic.BASIC.versionString);
        aboutForm.append(new StringItem("", "Copyright (c) 2003-2010, David Firth"));
        aboutForm.append(new StringItem("Build Date", BUILD_DATE));
        aboutForm.append(new StringItem("URL", "http://www.mobilebasic.com/"));
        
        String ticker = com.mobilebasic.BASIC.versionString + ", Build date " + BUILD_DATE + ", Visit http://www.mobilebasic.com/ for the latest Version";

        aboutForm.setTicker(new Ticker(ticker));

        // microedition.locale
        // microedition.encodingClass
        // microedition.http_proxy
        // microedition.encoding
        // microedition.platform
        // microedition.configuration
        // microedition.profiles

        aboutForm.addCommand(backCommand);
        aboutForm.setCommandListener(this);

        /*
         * Create Preferences Form
         */

        prefsForm = new Form("Preferences");

        fontSizeField = new ChoiceGroup("Font Size", List.EXCLUSIVE);
        fontSizeField.append("Automatic", null);
        fontSizeField.append("Small", null);
        fontSizeField.append("Medium", null);
        fontSizeField.append("Large", null);
        prefsForm.append(fontSizeField);

        textPauseField = new ChoiceGroup("Text pause", List.EXCLUSIVE);
        textPauseField.append("Disabled", null);
        textPauseField.append("Enabled", null);
        prefsForm.append(textPauseField);

        prefsForm.addCommand(backCommand);
        prefsForm.setCommandListener(this);
    }

    public Main()
    {
        display = Display.getDisplay(this); 

        exitCommand = new Command("Exit", Command.EXIT, 1);
        backCommand = new Command("Back", Command.BACK, 2);
        stopCommand = new Command("Stop", Command.ITEM, 3); 

        if (HELP_LIST)
          showHelpListCommand = new Command("Help", Command.ITEM, 4);

        loadCommand = new Command("Load", Command.ITEM, 7);
        enterCommand = new Command("Enter", Command.ITEM, 7);
        deleteCommand = new Command("Delete", Command.ITEM, 7);
        sizeCommand = new Command("Storage", Command.ITEM, 7);

        if (SAVE_COMMAND_ENABLED)
          saveCommand = new Command("Save", Command.ITEM, 7);

        prefsCommand = new Command("Preferences", Command.ITEM, 9);
        aboutCommand = new Command("About", Command.ITEM, 10);

        canvas = new BasicCanvas();
        canvas.addCommand(exitCommand);
        //canvas.addCommand(backCommand);
        //canvas.addCommand(aboutCommand);
        canvas.setCommandListener(this);

        gc = canvas.graphicsGc;

        randomAccessFile = new RandomAccessFile[MAXFILES];

        dataInput = new DataInput[MAXFILES];
        dataOutput = new DataOutput[MAXFILES];

        for (int i=0;i<MAXFILES;i++)
        {
          randomAccessFile[i] = null;
          dataInput[i] = null;
          dataOutput[i] = null;
        }

        InputStream inputStream;

        if (HELP_LIST)
        {
          inputStream = this.getClass().getResourceAsStream("/com/mobilebasic/Help.txt");
          if (inputStream != null)
            helpList = basic.ReadLines(new DataInputStream(inputStream));
        }

        exitList = new List("Exit BASIC?", List.IMPLICIT);
        exitList.append("No", null);
        exitList.append("Yes", null);
        exitList.addCommand(backCommand);
        exitList.setCommandListener(this);

        /*
         * Load Autorun Files embedded within the JAR file.
         */

        AutorunFlag = false;

        inputStream = this.getClass().getResourceAsStream("/Autorun.bas");
        if (inputStream != null)
        {
          try
          {
            Init(true);
            basic.LoadFrom(new DataInputStream(inputStream));
            AutorunFlag = true;
          }
          catch (Throwable t)
          {
            Init(false);
          }
        }
        else
        {
          inputStream = this.getClass().getResourceAsStream("/Autorun.lis");
          if (inputStream != null)
          {
            Init(true);
            boolean errorFlag = basic.Enter(new DataInputStream(inputStream));
            if (errorFlag == false)
              AutorunFlag = true;
            else
              Init(false);
          }
          else
          {
            Init(false);
          }
        }

        display.setCurrent(canvas);
    }

    private Form userForm = null;
    private int userExitStatus = 0;
    private Command userCancelCommand = null;
    private Command userProceedCommand = null;
    
    public void commandAction(Command command, Displayable screen)
    {
        if ((userForm != null) && (screen == userForm))
        {
            userExitStatus = (command == userProceedCommand) ? 1 : -1;
            
            synchronized(waitObject)
            {
                waitObject.notify();
            }
            return;
        }

        if (command == backCommand)
        {
            alert = null;

            if (screen == prefsForm)
            {
              canvas.SetFontSize(fontSizeField.getSelectedIndex());
              textPauseFlag = (textPauseField.getSelectedIndex() == 1);
            }

            if (alert == null)
            {
              display.setCurrent(canvas);
            }
            else
            {
              alert.setTimeout(Alert.FOREVER);
              display.setCurrent(alert, canvas);
            }
        }
        else if ((list != null) && (screen == list))
        {
            if (command == List.SELECT_COMMAND)
            {
                alert = null;

                int index = list.getSelectedIndex();
                if (index != -1)
                {
                    String text = list.getString(index);

                    if (text != null)
                    {
                        if ((listFunc == 1) || (listFunc == 2))
                        {
                            RandomAccessFile randomAccessFile = new RandomAccessFile(text, true, this);
                            try
                            {
                              if (listFunc == 1)
                              {
                                basic.LoadFrom(randomAccessFile);
                              }
                              else
                              {
                                basic.New();
                                basic.Enter(randomAccessFile);
                              }
                            }
                            catch (BasicError e)
                            {
                              alert = new Alert("Load error", e.getMessage(), null, AlertType.ERROR);
                            }
                            finally
                            {
                              try
                              {
                                randomAccessFile.close();
                              }
                              catch (IOException e)
                              {
                                alert = new Alert("Load error", e.getClass().getName(), null, AlertType.ERROR);
                              }
                            }
                        }
                        else if (listFunc == 3)
                        {
                            Delete(text);
                        }
                        else
                        {
                            for (int i=0;i<text.length();i++)
                                canvas.keyTyped((int)text.charAt(i));
                        }
                    }
                }

                list = null;

                if (alert == null)
                {
                  display.setCurrent(canvas);
                }
                else
                {
                  alert.setTimeout(Alert.FOREVER);
                  display.setCurrent(alert, canvas);
                }
            }
        }
        else if (screen == exitList)
        {
            if (command == List.SELECT_COMMAND)
            {
                int index = exitList.getSelectedIndex();
                if (index == 1)
                {
                    destroyApp(false); 
                    notifyDestroyed(); 
                }

                display.setCurrent(canvas);
            }
        }
        else
        {
            if (command == exitCommand)
            {
                exitList.setSelectedIndex(0, true);
                display.setCurrent(exitList);
            }
            else if (command == stopCommand)
            {
                basic.StopProgram();
            }
            else if (command == sizeCommand)
            {
              RecordStore recordStore = null;
              int diskSpace = 0;

              try
              {
                recordStore = RecordStore.openRecordStore(".CONFIG", false);
                diskSpace = recordStore.getSizeAvailable();
              }
              catch (Exception e)
              {
              }
              finally
              {
                try
                {
                  if (recordStore != null)
                    recordStore.closeRecordStore();
                }
                catch (Exception e)
                {
                }
              }

              alert = new Alert("Storage", "Free space: " + diskSpace + " byte(s)", null, AlertType.INFO);
              alert.setTimeout(Alert.FOREVER);
              display.setCurrent(alert, canvas);
            }
            else if (command == aboutCommand)
            {
                display.setCurrent(aboutForm);
            }
            else if (command == prefsCommand)
            {
                display.setCurrent(prefsForm);
            }
            else if ((command == loadCommand) ||
                     (command == enterCommand) ||
                     (command == deleteCommand))
            {
                String[] filename = RecordStore.listRecordStores();
                
                //list = new List("Load", List.IMPLICIT);
                list = new List(command.getLabel(), List.IMPLICIT);

                if (command == loadCommand)
                  listFunc = 1;
                else if (command == enterCommand)
                  listFunc = 2;
                else
                  listFunc = 3;
                
                if (filename != null)
                {
                    for (int i=0;i<filename.length;i++)
                    {
                        if (filename[i].charAt(0) != '.')
                            list.append(filename[i], null);
                    }
                }
                                
                list.addCommand(backCommand);
                list.setCommandListener(this);
                display.setCurrent(list);
            }
            else if (SAVE_COMMAND_ENABLED && (command == saveCommand))
            {
              String filename = EditForm("Save As", "Save", "Cancel", "Filename", "", 32, 0);
              if (filename != null)
              {
                  this.Delete(filename);
                  this.OpenFile(0, filename, false);
                  DataOutput dataOutput = this.GetDataOutputChannel(0);
                  if (dataOutput != null)
                    basic.SaveTo(dataOutput);
                  this.CloseFile(0);
              }
            }
            else if (HELP_LIST)
            {
                String title = null;
                Vector stringList = null;

                if (HELP_LIST)
                {
                  if (command == showHelpListCommand)
                  {
                    title = "Help";
                    stringList = helpList;
                  }
                }

                if (stringList != null)
                {
                    list = new List(title, List.IMPLICIT);
                    listFunc = 0;
                    for (int i=0;i<stringList.size();i++)
                        list.append((String)stringList.elementAt(i), null);
                    list.addCommand(backCommand);
                    list.setCommandListener(this);
                    display.setCurrent(list);
                }
            }
        }
    }
    
    private String waitObject = "X";

    private int UserForm(String title, String proceedText, String cancelText, Item item)
    {
        userForm = new Form(title);

        userForm.append(item);
        
        if (proceedText != null)
        {
            userProceedCommand = new Command(proceedText, Command.OK, 1);
            userForm.addCommand(userProceedCommand);
        }
        
        if (cancelText != null)
        {
            userCancelCommand = new Command(cancelText, Command.BACK, 1);
            userForm.addCommand(userCancelCommand);
        }

        userForm.setCommandListener(this);
        
        display.setCurrent(userForm);
        
        synchronized(waitObject)
        {
            try
            {
                waitObject.wait();
            }
            //catch (InterruptedException e)
            //{
            //}
            catch (Exception e)
            {
                //Class c = e.getClass();
                //System.out.println("Exception: " + c.getName());
            }
        }

        canvas.Focus();     // Tell canvas its getting the focus
        display.setCurrent(canvas);
        
        return userExitStatus;
    }

    public void run()
    {
        //System.out.println("Basic Thread: starting");
        
        if (AutorunFlag)
        {
          basic.parseLine("RUN", false);

          Bye();
        }

        PrintString(com.mobilebasic.BASIC.versionString + "\n");

        basic.New();
/*
        String[] program1 =
        {
            "1000 CLS",
            "1010 PLOT 5,5",
            "1020 DRAWLINE 5,5,25,25",
            "1030 SETCOLOR 255,0,0",
            "1040 FILLRECT 40,30,30,15",
            "1050 DRAWRECT 35,25,40,25",
            "1060 FILLROUNDRECT 30,20,50,35,10,10",
            "1070 DRAWROUNDRECT 25,15,60,45,10,10",
            "1080 SETCOLOR 0,0,255",
            "1090 FILLARC 30,30,50,50,60,-60",
            "1100 SETCOLOR 0,255,0",
            "1110 H$=\"X Press Fire2\"",
            "1120 XPOS%=(SCREENWIDTH(0)-STRINGWIDTH(H$))/2",
            "1130 YPOS%=(SCREENHEIGHT(0)-STRINGHEIGHT(H$))/2",
            "1140 DRAWSTRING H$,XPOS%,YPOS%",
            "1150 IF FIRE(0)=0 THEN GOTO 1150",
            "1160 BLIT 0,0,SCREENWIDTH(0),SCREENHEIGHT(0),SCREENWIDTH(0)/3,0",
            "1170 END",
            "RUN"
        };

        String[] program2 =
        {
            "1000 INPUT \"Principal: \",P",
            "1010 INPUT \"Rate %: \",R",
            "1020 INPUT \"Years: \",YEARS%",
            "1030 FOR N%=1 TO YEARS%",
            "1040 A=P*(1+R/100)^N%",
            "1050 PRINT \"YEAR: \" + STR$(N%) + \" AMOUNT: \" + STR$(A)",
            "1060 NEXT N%",
            "RUN"
        };

        String[] program3 =
        {
            "1000 A%=2-3",
            "1010 IF A%=-1 THEN PRINT \"Minus one\"",
            "1020 PRINT \"Finished\"",
            "RUN"
        };

        String[] program4 =
        {
            "1000 REM",
            "1010 REM ===================================================",
            "1020 REM Read Sections From DATA Statements and display menu",
            "1030 REM ===================================================",
            "1040 REM",
            "1050 RESTORE 5000",
            "1060 READ N%",
            "1070 DIM SECTIONS$(N%)",
            "1080 DIM LNO%(N%)",
            "1090 J%=N%-1",
            "1100 FOR I%=0 TO J%",
            "1110 READ T$",
            "1120 READ L%",
            "1130 SECTIONS$(I%) = T$",
            "1140 LNO%(I%)=L%",
            "1150 NEXT I%",
            "1160 SECTION%=CHOICEFORM(\"Converter\", \"Continue\", \"Exit\", \"Section\", SECTIONS$, 0)",
            "1170 IF SECTION%=-1 THEN END",
            "2000 REM",
            "2010 REM ===================================================================",
            "2020 REM Read available conversion within select section and display in menu",
            "2030 REM ===================================================================",
            "2040 REM",
            "2050 L%=LNO%(SECTION%)",
            "2060 RESTORE L%",
            "2070 READ N%",
            "2080 DIM TYPE$(N%)",
            "2090 DIM FACT(N%)",
            "2100 DIM DELTA(N%)",
            "2110 FACT(1)=1.2",
            "2120 FOR I%=1 TO N%",
            "2130 J%=I%-1",
            "2140 READ SVAL$",
            "2150 TYPE$(J%)=SVAL$",
            "2160 READ FVAL",
            "2170 FACT(J%)=FVAL",
            "2180 READ FVAL",
            "2190 DELTA(J%)=FVAL",
            "2200 NEXT I%",
            "2500 FROM%=CHOICEFORM(\"Converter\", \"Continue\", \"Back\", \"From\", TYPE$, 0)",
            "2505 IF FROM%=-1 THEN GOTO 1000",
            "2510 TO%=CHOICEFORM(\"Converter\", \"Continue\", \"Back\", \"To\", TYPE$, 0)",
            "2515 IF TO%=-1 THEN GOTO 2500",
            "2520 FROMLABEL$=TYPE$(FROM%)",
            "2530 FROMLABEL$=\"Enter \" + FROMLABEL$",
            "2535 FROM$=\"1.0\"",
            "2540 I%=EDITFORM(\"Converter\", \"Continue\", \"Back\", FROMLABEL$, FROM$, 10, 0)",
            "2550 IF I%=-1 THEN GOTO 2510",
            "2560 TRAP 2590",
            "2570 FROM=VAL(FROM$)",
            "2580 GOTO 2610",
            "2590 IVAL%=VAL(FROM$)",
            "2600 FROM=IVAL%",
            "2610 MULT=FACT(FROM%)",
            "2620 PLUS=DELTA(FROM%)",
            "2630 DIV=FACT(TO%)",
            "2640 SUB=DELTA(TO%)",
            "2650 FVAL=FROM+PLUS",
            "2660 FVAL=FVAL*MULT",
            "2670 FVAL=FVAL/DIV",
            "2680 FVAL=FVAL-SUB",
            "2690 VAL$=STR$(FVAL)",
            "2700 MSG$=TYPE$(TO%)",
            "2710 MSG$=\" \" + MSG$",
            "2720 MSG$=VAL$+MSG$",
            "2730 I%=MESSAGEFORM(\"Converter\", \"Continue\", \"Exit\", \"Result: \", MSG$)",
            "2740 IF I%=-1 THEN END",
            "2750 GOTO 2540",
            "5000 REM",
            "5010 REM =============================",
            "5020 REM DATA FOR MAIN CONVERSION MENU",
            "5030 REM =============================",
            "5040 REM",
            "5050 DATA 6",
            "5060 DATA Length,6000",
            "5070 DATA Weight,6200",
            "5080 DATA Fuel Consumption,6400",
            "5090 DATA Temperature,6600",
            "5100 DATA Power,6800",
            "5110 DATA Volume,7000",
            "6000 DATA 7",
            "6010 DATA Inches,25.4,0.0",
            "6020 DATA Feet,305.0,0.0",
            "6030 DATA Miles,1609000.0,0.0",
            "6040 DATA Millimetres,1.0,0.0",
            "6050 DATA Centimetres,10.0,0.0",
            "6060 DATA Metres,1000.0,0.0",
            "6070 DATA Kilometres,1000000.0,0.0",
            "6210 DATA 4",
            "6220 DATA Ounces(oz),28.35,0.0",
            "6230 DATA Pounds(lb),454.0,0.0",
            "6240 DATA Grams(g),1.0,0.0",
            "6250 DATA Kilograms(kg),1000.0,0.0",
            "6400 DATA 2",
            "6410 DATA mpg,0.354,0.0",
            "6420 DATA km/l,1.0,0.0",
            "6600 DATA 3",
            "6610 DATA Fahrenheit,0.56,-32.0",
            "6620 DATA Centigrade,1.0,0.0",
            "6630 DATA Celsius,1.0,0.0",
            "6800 DATA 2",
            "6810 DATA Horsepower(hp),745.7,0.0",
            "6820 DATA Watts(W),1.0,0.0",
            "7000 DATA 6",
            "7010 DATA Litres(l),1.0,0.0",
            "7020 DATA Gallons(Imp gal),4.546,0.0",
            "7030 DATA US gallons(US gal),3.785,0.0",
            "7040 DATA Pints(Imp pt),0.568,0.0",
            "7050 DATA Quarts(Imp qt),1.137,0.0",
            "7060 DATA Quarts(US qt),0.946,0.0",
            "RUN",
        };

        String[] program5 =
        {
            "1000 DIM A$(3)",
            "1010 A$(0)=\"Alloy Wheels\"",
            "1020 A$(1)=\"Leather\"",
            "1030 A$(2)=\"CD Player\"",
            "1040 I%=CHOICEFORM(\"Choice Form\", \"Continue\", \"Cancel\", \"My Choice\", A$, 0)",
            "1050 PRINT I%",
            "1060 I%=CHOICEFORM(\"Options\", \"Continue\", \"Cancel\", \"Optional Extras\", A$, 1)",
            "1070 PRINT I%",

            "2000 DIM J%(2)",
            "2010 J%(0)=DAYS(0)",
            "2020 J%(1)=MILLISECONDS(0)",
            "2030 I%=DATEFORM(\"Date Form\", \"Continue\", \"Cancel\", \"Today\", J%, 0)",
            "2035 PRINT I%",
            "2040 I%=DATEFORM(\"Date Form2\", \"Continue\", \"Cancel\", \"My Date2\", J%, 0)",
            "2050 PRINT I%",

            "3000 T$=\"Default Text\"",
            "3010 I%=EDITFORM(\"Edit Form\", \"Continue\", \"Cancel\", \"Description\", T$, 40, 0)",
            "3020 PRINT STR$(I%) + \": \" + T$",
            "3030 PRINT I%",
            "3040 I%=EDITFORM(\"Edit Form\", \"Continue\", \"Cancel\", \"Password\", T$, 40, 1)",
            "3050 PRINT T$",
            "3060 PRINT I%",
            "3070 T$=\"1234\"",
            "3080 I%=EDITFORM(\"Edit Form\", \"Continue\", \"Cancel\", \"Numeric\", T$, 40, 2)",
            "3090 PRINT T$",
            "3100 PRINT I%",

            "4000 I% = 5",
            "4010 I%=GAUGEFORM(\"Gauge Form\", \"Continue\", \"Cancel\", \"Volume\", 10, I%, 1)",
            "4020 PRINT I%",
            "4030 IF I%<>-1 THEN GOTO 4010",

            "5000 I%=MESSAGEFORM(\"Message Form\", \"Continue\", \"Cancel\", \"Info: \", \"Mobile BASIC contains several build-in forms including:- CHOICEFORM(), EDITFORM(), DATEFORM(), GAUGEFORM() and MESSAGEFOR()\")",
            "5010 PRINT I%",
            "RUN"
        };

        String[] program6 =
        {
            "1000 OPEN #0,\"test2.dat\",\"OUTPUT\"",
            "1010 FOR I%=0 TO 3",
            "1030 PUT #0,I%",
            "1040 PRINT #0,123456789",
            "1050 PRINT #0,1.234*I%",
            "1060 PRINT #0,\"Hello World\"",
            "1070 NEXT I%",
            "1080 CLOSE #0",

            "2000 OPEN #1,\"test2.dat\",\"INPUT\"",
            "2010 TRAP 3000",
            "2020 GET #1,I%",
            "2030 INPUT #1,J%",
            "2040 INPUT #1,F",
            "2050 INPUT #1,S$",
            "2060 PRINT I%",
            "2070 PRINT J%",
            "2080 PRINT F",
            "2090 PRINT S$",
            "2100 GOTO 2020",

            "3000 PRINT \"EOF\"",
            "3010 CLOSE #1",

            "DELETE \"test2.dat\"",
            "DIR",
            "RUN"
        };

        String[] program7 =
        {
            "10 I%=3",
            "20 I%=I%*100",
            "30 GOTO I%",
            "40 END",
            "100 PRINT 100",
            "110 END",
            "200 PRINT 200",
            "210 END",
            "300 PRINT 300",
            "310 END",
            "RUN"
        };

        String[] program8 =
        {
            "10 GOSUB 100",
            "20 PRINT \"20\"",
            "30 END",
            "100 GOSUB 200",
            "110 PRINT \"110\"",
            "120 END",
            "200 GOSUB 300",
            "210 PRINT \"210\"",
            "220 END",
            "300 GOSUB 400",
            "310 PRINT \"310\"",
            "320 END",
            "400 PRINT \"400\"",
            "401 POP: REM 210",
            "402 POP: REM 110",
            "403 POP: REM 20",
            "404 POP: REM ERR ON RETURN",
            "410 RETURN",
            "RUN"
        };

        String[] program9 =
        {
            "10 CLS",
            "20 PLOT 5,5",
            "30 DRAWLINE 5,5,25,25*3",
            "35 SETCOLOR 255,0,0",
            "40 FILLRECT 40,30,30,15",
            "50 DRAWRECT 35,25,40,25",
            "60 FILLROUNDRECT 30,20,50,35,10,10",
            "70 DRAWROUNDRECT 25,15,60,45,10,10",
            "74 SETCOLOR 0,0,255",
            "75 ANG%=-60",
            "80 FILLARC 30,30,50,50,60,ANG%",
            "90 FILLARC 30,30,50,50,60,-60",
            "100 SETCOLOR 0,255,0",
            "110 H$=\"Press Fire\"",
            "120 SCREENWIDTH%=SCREENWIDTH(0)",
            "130 SCREENHEIGHT%=SCREENHEIGHT(0)",
            "140 STRINGWIDTH%=STRINGWIDTH(H$)",
            "150 STRINGHEIGHT%=STRINGHEIGHT(H$)",
            "160 XPOS%=SCREENWIDTH%-STRINGWIDTH%",
            "170 YPOS%=SCREENHEIGHT%-STRINGHEIGHT%",
            "180 DRAWSTRING H$,XPOS%,YPOS%",
            "190 A%=FIRE(0)",
            "200 IF A%=0 THEN GOTO 190",
            "210 BLIT 0,0,SCREENWIDTH%,SCREENHEIGHT%,20,0",
            "220 END",
            "RUN"
        };

        String[] program10 =
        {
            "10 PRINT SIN(3)",
            "20 DEG",
            "30 PRINT SIN(45)",
            "40 PRINT SIN(45.0)",
            "50 PRINT SQR(81.0)",
            "60 PRINT SQR(0.0)",
            "RUN"
        };

        String[] program11 =
        {
            "1000 PRINT \"SQR(81.0)=\" + STR$(SQR(81.0))",
            "1010 PRINT \"3.0 ^ 2.0=\" + STR$(3.0 ^ 2.0)",
            "1020 PRINT \"LOG(97.0)=\" + STR$(LOG(97.0))",
            "1030 PRINT \"EXP(4.317488)\" + STR$(EXP(4.317488))",
            "1040 DEG",
            "1050 SIN45=SIN(45.0)",
            "1060 COS45=COS(45.0)",
            "1070 TAN45=TAN(45.0)",
            "1080 PRINT \"ASIN(\" + STR$(SIN45) + \")=\" + STR$(ASIN(SIN45))",
            "1090 PRINT \"ACOS(\" + STR$(COS45) + \")=\" + STR$(ACOS(COS45))",
            "1100 PRINT \"ATAN(\" + STR$(TAN45) + \")=\" + STR$(ATAN(TAN45))",
            "1110 RAD",
            "1120 ANG30=30/57.29578",
            "1130 PRINT ANG30",
            "1140 SIN30=SIN(ANG30)",
            "1150 COS30=COS(ANG30)",
            "1160 TAN30=TAN(ANG30)",
            "1170 PRINT \"ASIN(\" + STR$(SIN30) + \")=\" + STR$(ASIN(SIN30))",
            "1180 PRINT \"ACOS(\" + STR$(COS30) + \")=\" + STR$(ACOS(COS30))",
            "1190 PRINT \"ATAN(\" + STR$(TAN30) + \")=\" + STR$(ATAN(TAN30))",
            "RUN"
        };

        String[] program12 =
        {
            "100 LIST",
            "100 DELETE \"test.dat\"",
            "1000 OPEN #1,\"test.dat\",\"output\"",
            "1010 NOTE #1,N1%",
            "1020 PRINT #1,\"Hello\"",
            "1030 NOTE #1,N2%",
            "1040 PRINT #1,\"Big\"",
            "1050 NOTE #1,N3%",
            "1060 PRINT #1,\"World\"",
            "1070 CLOSE #1",
            "1071 PRINT N1%",
            "1072 PRINT N2%",
            "1073 PRINT N3%",
            "1080 OPEN #1,\"test.dat\",\"input\"",
            "1090 POINT #1,N3%",
            "1100 INPUT #1,A$",
            "1110 CLOSE #1",
            "1120 PRINT A$",
            "3000 OPEN #1,\"test.dat\",\"input\"",
            "3020 TRAP 3060",
            "3030 INPUT #1,A$",
            "3040 PRINT \"> \" + A$",
            "3050 GOTO 3030",
            "3060 CLOSE #1",
            "4000 TRAP -1",
            "RUN"
         };

        String[] program13 =
        {
            "10 A%=1+2",
            "20 PRINT A%",
            "30 B=3.4E+1+4.6E+1",
            "40 PRINT B",
            "RUN"
        };

        String[] program14 =
        {
            "10 DIM A%(10+6)",
            "20 A%(5+3*3)=3*10-4*2:A%(13)=1313:A%(15)=1515",
            "30 PRINT STR$(A%(13))+\" \"+STR$(A%(14))+\" \"+STR$(A%(15))",
            "RUN"
        };

        String[] program15 =
        {
            "10 A%=5+(4*3):PRINT A%-2*7:PRINT FRE(0)",
            "20 X$=\"HELLO WORLD\"",
            "30 PRINT X$",
            "40 PRINT LEFT$(X$,5)",
            "50 PRINT RIGHT$(X$,5)",
            "60 PRINT MID$(X$,5,3)",
            "70 PRINT RIGHT$(X$,5)+\" \"+LEFT$(X$,5)",
            "80 PRINT CHR$(65)+CHR$(66)",
            "90 PRINT ASC(MID$(X$,2,1))",
            "100 PRINT \"X=\" + STR$(3.142)",
            "110 PRINT \"X=\" + STR$(3)",
            "120 PRINT LEN(X$)",
            "130 PRINT VAL(\"3.142\")",
            "140 PRINT VAL(\"3\")",
            "150 I%=VAL(\"3.142\")",
            "160 FP=VAL(\"3.142\")",
            "170 PRINT STR$(I%)+\" \"+STR$(FP)",
            "RUN"
        };

        String[] program16 =
        {
            "10 INPUT \"String: \",S$",
            "20 PRINT S$",
            "30 INPUT \"Integer: \",I%",
            "40 PRINT I%",
            "50 INPUT \"Float: \",F",
            "60 PRINT F",
            "RUN"
        };

        String[] program17 =
        {
            "10 PRINT \"Starting\"",
            "20 Z%=3",
            "30 J%=4",
            "40 FOR X%=Z%*J% TO 10*10 STEP 2*3",
            "50 PRINT X%",
            "60 NEXT X%",
            "70 PRINT \"Finished\"",
            "RUN"
        };

        String[] program18 =
        {
            "10 I%=1",
            "20 IF I%=1 THEN PRINT \"Hello\":PRINT \"World\"",
            "30 PRINT \"Goodbye\"",
            "LIST 10,20",
            "RUN"
        };

        String[] program19 =
        {
            "10 REM",
            "20 REM   xyz abc",
            "30 DATA 111,222,333",
            "40 DIM I%(3)",
            "50 FOR J%=2 TO 0 STEP -1",
            "60 READ I%(J%)",
            "70 NEXT J%",
            "80 FOR J%=0 TO 2",
            "90 PRINT I%(J%)",
            "100 NEXT J%",
            "RUN"
        };

        String[] program20 =
        {
            "10 DAYS%=DAYS(0)",
            "20 MS%=MILLISECONDS(0)",
            "30 PRINT DAYS%",
            "40 PRINT MS%",
            "50 Y%=YEAR(DAYS%, MS%)",
            "60 M%=MONTH(DAYS%, MS%)",
            "70 D%=DAY(DAYS%, MS%)",
            "80 PRINT STR$(D%)+\"/\"+STR$(M%)+\"/\"+STR$(Y%)",
            "90 H%=HOUR(DAYS%, MS%)",
            "100 M%=MINUTE(DAYS%, MS%)",
            "110 S%=SECOND(DAYS%, MS%)",
            "120 MS%=MILLISECOND(DAYS%, MS%)",
            "130 PRINT STR$(H%)+\":\"+STR$(M%)+\":\"+STR$(S%)+\".\"+STR$(MS%)",
            "RUN"
        };

        String[] program = program9;

        for (int i=0; i<program.length; i++)
        {
            String line = program[i];
            basic.parseLine(line, false);
        }
*/
        while (true)
        {
            canvas.addCommand(aboutCommand);

            if (HELP_LIST)
              canvas.addCommand(showHelpListCommand);

            canvas.addCommand(loadCommand);
            canvas.addCommand(enterCommand);
            canvas.addCommand(deleteCommand);
            canvas.addCommand(sizeCommand);

            if (SAVE_COMMAND_ENABLED)
              canvas.addCommand(saveCommand);

            canvas.addCommand(prefsCommand);

            canvas.printString("READY\n", 0, -1, true, textPauseFlag);
            
            String inputLine = canvas.GetLine("", null);

            if (HELP_LIST)
              canvas.removeCommand(showHelpListCommand);

            canvas.removeCommand(aboutCommand);
            canvas.removeCommand(loadCommand);
            canvas.removeCommand(enterCommand);
            canvas.removeCommand(deleteCommand);
            canvas.removeCommand(sizeCommand);

            if (SAVE_COMMAND_ENABLED)
              canvas.removeCommand(saveCommand);

            canvas.removeCommand(prefsCommand);

            canvas.addCommand(stopCommand);

            canvas.Init();
            basic.parseLine(inputLine, true);
            canvas.Init();            
            
            canvas.removeCommand(stopCommand);
        }
    }
    
    /*
     * BASIC Support Routines
     */

    public void Message(String s)
    {
      PrintString(s + "\n");
    }

    public void Error(String s)
    {
      PrintString(s + "\n");
    }

  private static final int MAXFILES = 10;
  private RandomAccessFile[] randomAccessFile;
  private DataInput[] dataInput;
  private DataOutput[] dataOutput;

  public void OpenFile(int iocb, String filename, boolean readOnlyFlag)
  {
    if ((iocb >= 0) && (iocb < MAXFILES))
    {
      boolean channelFree = false;

      if (randomAccessFile[iocb] == null)
        channelFree = true;

      if (channelFree)
      {
          try
          {
            randomAccessFile[iocb] = new RandomAccessFile(filename, readOnlyFlag, this);
            dataInput[iocb] = randomAccessFile[iocb];
            dataOutput[iocb] = randomAccessFile[iocb];
          }
          catch (Exception e)
          {
            randomAccessFile[iocb] = null;
            dataInput[iocb] = null;
            dataOutput[iocb] = null;
            throw new BasicError(BasicError.IO_ERROR, e.getClass().getName());
          }
      }
      else
      {
        throw new BasicError(BasicError.CHANNEL_ALREADY_IN_USE, "Channel " + iocb + " already in use");
      }
    }
    else
    {
      throw new BasicError(BasicError.INVALID_CHANNEL, "Invalid channel");
    }
  }

  public void CloseFile(int iocb)
  {
    if ((iocb >= 0) && (iocb < MAXFILES))
    {
      try
      {
        if (randomAccessFile[iocb] != null)
          randomAccessFile[iocb].close();
      }
      catch (IOException e)
      {
        throw new BasicError(BasicError.IO_ERROR, e.getMessage());
      }
      finally
      {
        randomAccessFile[iocb] = null;
        dataInput[iocb] = null;
        dataOutput[iocb] = null;
      }
    }
    else
    {
      throw new BasicError(BasicError.INVALID_CHANNEL, "Invalid channel");
    }
  }

  public void CloseAllFiles()
  {
    for (int i=0;i<MAXFILES;i++)
      CloseFile(i);
  }

  public int Note(int iocb)
  {
    long offset = -1;

    if ((iocb >= 0) && (iocb < MAXFILES))
    {
      offset  = randomAccessFile[iocb].getFilePointer();
    }
    else
    {
      throw new BasicError(BasicError.INVALID_CHANNEL, "Invalid channel");
    }

    return (int)offset;
  }

  public void Point(int iocb, int offset)
  {
    if ((iocb >= 0) && (iocb < MAXFILES))
    {
      randomAccessFile[iocb].seek(offset);
    }
    else
    {
      throw new BasicError(BasicError.INVALID_CHANNEL, "Invalid channel");
    }
  }

  public DataInput GetDataInputChannel(int iocb)
  {
    if ((iocb >= 0) && (iocb < MAXFILES))
    {
      return dataInput[iocb];
    }
    else
    {
      throw new BasicError(BasicError.INVALID_CHANNEL, "Invalid channel");
    }
  }

  public DataOutput GetDataOutputChannel(int iocb)
  {
    if ((iocb >= 0) && (iocb < MAXFILES))
    {
      return dataOutput[iocb];
    }
    else
    {
      throw new BasicError(BasicError.INVALID_CHANNEL, "Invalid channel");
    }
  }

  public void PutByte (int iocb, int byteValue)
  {
    if ((iocb >= 0) && (iocb < MAXFILES))
    {
      try
      {
        dataOutput[iocb].writeByte(byteValue);
      }
      catch (IOException e)
      {
        Class c = e.getClass();
        throw new BasicError(BasicError.IO_ERROR, c.getName());
      }
      catch (NullPointerException e)
      {
        throw new BasicError(BasicError.IO_ERROR, "Channel not writable");
      }
    }
    else
    {
      throw new BasicError(BasicError.INVALID_CHANNEL, "Invalid channel");
    }
  }

  public void PutInt (int iocb, int intValue)
  {
    if ((iocb >= 0) && (iocb < MAXFILES))
    {
      try
      {
        dataOutput[iocb].writeInt(intValue);
      }
      catch (IOException e)
      {
        Class c = e.getClass();
        throw new BasicError(BasicError.IO_ERROR, c.getName());
      }
      catch (NullPointerException e)
      {
        throw new BasicError(BasicError.IO_ERROR, "Channel not writable");
      }
    }
    else
    {
      throw new BasicError(BasicError.INVALID_CHANNEL, "Invalid channel");
    }
  }

  public void PutString(int iocb, String s)
  {
    if ((iocb >= 0) && (iocb < MAXFILES))
    {
      try
      {
        dataOutput[iocb].writeUTF(s);
      }
      catch (IOException e)
      {
        Class c = e.getClass();
        throw new BasicError(BasicError.IO_ERROR, c.getName());
      }
      catch (NullPointerException e)
      {
        throw new BasicError(BasicError.IO_ERROR, "Channel not writable");
      }
    }
    else
    {
      throw new BasicError(BasicError.INVALID_CHANNEL, "Invalid channel");
    }
  }

  public int GetByte (int iocb)
  {
    int byteValue;

    if ((iocb >= 0) && (iocb < MAXFILES))
    {
      try
      {
        byteValue = dataInput[iocb].readByte();
      }
      catch (IOException e)
      {
        Class c = e.getClass();
        throw new BasicError(BasicError.IO_ERROR, c.getName());
      }
      catch (NullPointerException e)
      {
        throw new BasicError(BasicError.IO_ERROR, "Channel not readable");
      }
    }
    else
    {
      throw new BasicError(BasicError.INVALID_CHANNEL, "Invalid channel");
    }

    return byteValue;
  }

  public int GetInt (int iocb)
  {
    int intValue;

    if ((iocb >= 0) && (iocb < MAXFILES))
    {
      try
      {
        intValue = dataInput[iocb].readInt();
      }
      catch (IOException e)
      {
        Class c = e.getClass();
        throw new BasicError(BasicError.IO_ERROR, c.getName());
      }
      catch (NullPointerException e)
      {
        throw new BasicError(BasicError.IO_ERROR, "Channel not readable");
      }
    }
    else
    {
      throw new BasicError(BasicError.INVALID_CHANNEL, "Invalid channel");
    }

    return intValue;
  }

  public String GetString (int iocb)
  {
    String s;

    if ((iocb >= 0) && (iocb < MAXFILES))
    {
      try
      {
        s = dataInput[iocb].readUTF();
      }
      catch (IOException e)
      {
        Class c = e.getClass();
        throw new BasicError(BasicError.IO_ERROR, c.getName());
      }
      catch (NullPointerException e)
      {
        throw new BasicError(BasicError.IO_ERROR, "Channel not readable");
      }
    }
    else
    {
      throw new BasicError(BasicError.INVALID_CHANNEL, "Invalid channel");
    }

    return s;
  }

    public void PrintString(String s)
    {
        canvas.printString(s, 0, -1, true, textPauseFlag);
    }
    
    public void CLS()
    {
        gc.setColor(0xffffff);
        gc.fillRect(0, 0, canvas.getWidth(), canvas.getHeight());
        gc.setColor(0x000000);
        canvas.xposText = 0;
        canvas.yposText = 0;
    }
    
    public void DrawLine(int fromX, int fromY, int toX, int toY)
    {
        gc.drawLine (fromX, fromY, toX, toY);
    }
    
    public void FillRect(int x, int y, int w, int h)
    {
        gc.fillRect(x, y, w, h);
    }

    public void DrawRect(int x, int y, int w, int h)
    {
        gc.drawRect(x, y, w, h);
    }

    public void FillRoundRect(int x, int y, int w, int h, int arcWidth, int arcHeight)
    {
        gc.fillRoundRect(x, y, w, h, arcWidth, arcHeight);
    }

    public void DrawRoundRect(int x, int y, int w, int h, int arcWidth, int arcHeight)
    {
        gc.drawRoundRect(x, y, w, h, arcWidth, arcHeight);
    }
    
    public void FillArc(int x, int y, int w, int h, int startAngle, int arcAngle)
    {
        gc.fillArc(x, y, w, h, startAngle, arcAngle);
    }

    public void DrawArc(int x, int y, int w, int h, int startAngle, int arcAngle)
    {
        gc.drawArc(x, y, w, h, startAngle, arcAngle);
    }
    
    public void SetColor(int r, int g, int b)
    {
        gc.setColor(r, g, b);
    }
    
    public void Blit(int fromX, int fromY, int w, int h, int toX, int toY)
    {
        canvas.Blit(fromX, fromY, w, h, toX, toY);
    }

    public void GelLoad(String gelName, String resourceName)
    {
      canvas.GelLoad(gelName, resourceName);
    }

    public void GelGrab(String gelName, int x, int y, int w, int h)
    {
      canvas.GelGrab(gelName, x, y, w, h);
    }

    public int GelWidth(String gelName)
    {
      return canvas.GelWidth(gelName);
    }

    public int GelHeight(String gelName)
    {
      return canvas.GelHeight(gelName);
    }

    public void DrawGel(String gelName, int x, int y)
    {
      canvas.DrawGel(gelName, x, y);
    }

    public void SpriteGEL(String spriteName, String gelName)
    {
      canvas.SpriteGEL(spriteName, gelName);
    }

    public void SpriteMove(String spriteName, int x, int y)
    {
        canvas.SpriteMove(spriteName, x, y);
    }

    public int SpriteHit(String spriteName1, String spriteName2)
    {
        return canvas.SpriteHit(spriteName1, spriteName2);
    }
    
    public void DrawString(String s, int x, int y)
    {
        gc.drawString(s, x, y, Graphics.TOP | Graphics.LEFT);
    }
    
    public int ScreenWidth()
    {
        return canvas.getWidth();
    }
    
    public int ScreenHeight()
    {
        return canvas.getHeight();
    }
    
    public int isColor()
    {
        return display.isColor() ? 1 : 0;
    }
    
    public int NumColors()
    {
        return display.numColors();
    }
    
    public int StringWidth(String s)
    {
        return canvas.font.stringWidth(s);
    }

    public int StringHeight(String s)
    {
        return canvas.font.getHeight();
    }
    
    public int Up()
    {
        return (canvas.gameActionBits & BasicCanvas.GAME_UP);
    }
    
    public int Down()
    {
        return (canvas.gameActionBits & BasicCanvas.GAME_DOWN);
    }
    
    public int Left()
    {
        return (canvas.gameActionBits & BasicCanvas.GAME_LEFT);
    }
    
    public int Right()
    {
        return (canvas.gameActionBits & BasicCanvas.GAME_RIGHT);
    }
    
    public int Fire()
    {
        return (canvas.gameActionBits & BasicCanvas.GAME_FIRE);
    }
    
    public int GameA()
    {
        return (canvas.gameActionBits & BasicCanvas.GAME_A);
    }

    public int GameB()
    {
        return (canvas.gameActionBits & BasicCanvas.GAME_B);
    }
    
    public int GameC()
    {
        return (canvas.gameActionBits & BasicCanvas.GAME_C);
    }
    
    public int GameD()
    {
        return (canvas.gameActionBits & BasicCanvas.GAME_D);
    }
        
    public int Year(Date date)
    {
        calendar.setTime(date);
        return calendar.get(Calendar.YEAR);
    }
    
    public int Month(Date date)
    {
        calendar.setTime(date);
        return calendar.get(Calendar.MONTH) + 1;
    }
    
    public int Day(Date date)
    {
        calendar.setTime(date);
        return calendar.get(Calendar.DAY_OF_MONTH);
    }

    public int Hour(Date date)
    {               
        calendar.setTime(date);
        return calendar.get(Calendar.HOUR_OF_DAY);
    }
    
    public int Minute(Date date)
    {
        calendar.setTime(date);
        return calendar.get(Calendar.MINUTE);
    }
    
    public int Second(Date date)
    {
        calendar.setTime(date);
        return calendar.get(Calendar.SECOND);
    }
    
    public int Millisecond(Date date)
    {
        calendar.setTime(date);
        return calendar.get(Calendar.MILLISECOND);
    }
    
    public Enumeration Directory(String filter)
    {
      String[] filenames = RecordStore.listRecordStores();
      Vector vector = null;
      Enumeration dirEnum = null;

      if (filenames != null)
      {
        for (int i=0;i<filenames.length;i++)
        {
          String filename = filenames[i];

          if (filename.charAt(0) != '.')
          {
            boolean match = true;

            char filterCh = (char)0x0000;
            int fOffset = 0;

            for (int j=0;j<filename.length();j++)
            {
              if (filterCh == (char)0x0000)
              {
                if (fOffset < filter.length())
                {
                  filterCh = filter.charAt(fOffset++);
                }
                else
                {
                  match = false;
                  break;
                }
              }

              if (filterCh == '*')
              {
                /*
                 * Check next filter character (if present)
                 * for wildcard termination character
                 */

                if (fOffset < filter.length())
                {
                  char nextFilterCh = filter.charAt(fOffset);
                  if (filename.charAt(j) == nextFilterCh)
                  {
                    filterCh = (char)0x0000;	// Wildcard finished
                    fOffset++; // Move past character following '*'
                  }
                }
              }
              else if (filename.charAt(j) == filterCh)
              {
                filterCh = (char)0x0000;	// Filter character used
              }
              else
              {
                match = false;
                break;
              }
            }

            if (fOffset != filter.length())
              match = false;

            if (match)
            {
              if (vector == null)
                vector = new Vector();
              vector.addElement(filename);
            }
          }
        }
      }

      if (vector != null)
        dirEnum = vector.elements();

      return dirEnum;
    }
    
    public String GetLine(String prompt, String defaultText)
    {
        Displayable originalDisplayable = display.getCurrent();
        
        if (originalDisplayable != canvas)
            display.setCurrent(canvas);

        String line = canvas.GetLine(prompt, defaultText);
        
        if (originalDisplayable != canvas)
            display.setCurrent(originalDisplayable);
        
        return line;
    }
    
    public void Bye()
    {
        destroyApp(false);
        notifyDestroyed();
    }
    
    public void Delete(String filename)
    {
        try
        {
            RecordStore.deleteRecordStore(filename);
        }
        catch (RecordStoreNotFoundException e)
        {
        }
        catch (RecordStoreException e)
        {
            //e.printStackTrace();
        }
    }

    public String EditForm(String formTitle, String proceedText, String cancelText, String label, String defaultText, int maxLen, int mode)
    {
        String text = null;
        
        switch (mode)
        {
            case 0 :
                mode = TextField.ANY;
                break;
            case 1 :
                mode = TextField.PASSWORD;
                break;
            case 2 :
                mode = TextField.NUMERIC;
                break;
            case 3 :
                mode = TextField.EMAILADDR;
                break;
            case 4 :
                mode = TextField.PHONENUMBER;
                break;
            case 5 :
                mode = TextField.URL;
                break;
            default :
                throw new BasicError(BasicError.VALUE_ERROR, "type must be 0..5");
        }

        if (maxLen > 0)
        {
            try
            {
                TextField textField = new TextField(label, defaultText, maxLen, mode);

                int res = UserForm(formTitle, proceedText, cancelText, textField);
                if (res == 1)
                    text = textField.getString();
            }
            catch (IllegalArgumentException e)
            {
                throw new BasicError(BasicError.VALUE_ERROR, "Invalid default text");
            }
        }
        else
        {
            throw new BasicError(BasicError.VALUE_ERROR, "Maximum length must be >0");
        }
        
        return text;
    }
    
    public Date DateForm(String formTitle, String proceedText, String cancelText, String label, Date date, int mode)
    {                            
        if (mode == 1)
            mode = DateField.DATE;
        else if (mode == 2)
            mode = DateField.TIME;
        else
            mode = DateField.DATE_TIME;
        
        DateField dateField = new DateField(label, mode);
        if (date != null)
            dateField.setDate(date);

        int res = UserForm(formTitle, proceedText, cancelText, dateField);
        if (res == 1)
            date = dateField.getDate();
        else
            date = null;
        
        return date;
    }
    
    public int ChoiceForm(String formTitle, String proceedText, String cancelText, String label, String[] stringArray, int mode)
    {
        int res = -1;
        
        if (mode == 0)
        {
            mode = Choice.EXCLUSIVE;
        }
        else
        {
            mode = Choice.MULTIPLE;
            if (stringArray.length > 32)
                throw new BasicError(BasicError.VALUE_ERROR, "Maximum of 32 items in a multiple choice");
        }
                                
        ChoiceGroup choiceGroup = new ChoiceGroup(label, mode, stringArray, null);
        res = UserForm(formTitle, proceedText, cancelText, choiceGroup);
        if (res == 1)
        {
            if (mode == Choice.MULTIPLE)
            {
                boolean selectedArray[] = new boolean[32];
                choiceGroup.getSelectedFlags(selectedArray);
                res = 0;
                for (int i=31;i>=0;i--)
                {
                    res = res << 1;
                    if (selectedArray[i])
                        res |= 1;
                }
            }
            else
            {
                res = choiceGroup.getSelectedIndex();
            }   
        }
        else
        {
            res = -1;
        }

        return res;
    }

    public int GaugeForm(String formTitle, String proceedText, String cancelText, String label, int maxValue, int initialValue, int mode)
    {
        int res = -1;
        
        if (maxValue > 0)
        {
            Gauge gauge = new Gauge(label, (mode == 1), maxValue, initialValue);
            res = UserForm(formTitle, proceedText, cancelText, gauge);
            if (res == 1)
                res = gauge.getValue();
            else
                res = -1;
        }
        else
        {
            throw new BasicError(BasicError.VALUE_ERROR, "Maximum value must be >0");
        }
        
        return res;
    }

    public int MessageForm(String formTitle, String proceedText, String cancelText, String label, String msg)
    {
        StringItem stringItem = new StringItem(label, msg);

        return UserForm(formTitle, proceedText, cancelText, stringItem);
    }
}
