/*
	Реализация спецификаций CLDC версии 1.1 (JSR-139), MIDP версии 2.1 (JSR-118)
	и других спецификаций для функционирования компактных приложений на языке
	Java (мидлетов) в среде программного обеспечения Малик Эмулятор.

	Copyright © 2016, 2019 Малик Разработчик

	Это свободная программа: вы можете перераспространять ее и/или изменять
	ее на условиях Меньшей Стандартной общественной лицензии GNU в том виде,
	в каком она была опубликована Фондом свободного программного обеспечения;
	либо версии 3 лицензии, либо (по вашему выбору) любой более поздней версии.

	Эта программа распространяется в надежде, что она будет полезной,
	но БЕЗО ВСЯКИХ ГАРАНТИЙ; даже без неявной гарантии ТОВАРНОГО ВИДА
	или ПРИГОДНОСТИ ДЛЯ ОПРЕДЕЛЕННЫХ ЦЕЛЕЙ. Подробнее см. в Меньшей Стандартной
	общественной лицензии GNU.

	Вы должны были получить копию Меньшей Стандартной общественной лицензии GNU
	вместе с этой программой. Если это не так, см.
	<https://www.gnu.org/licenses/>.
*/


package malik.emulator.media.sound;

import malik.emulator.application.*;
import malik.emulator.util.*;

public final class SoundPlayerSynthetic extends Object
		implements PlayerMIDIEndTrackHandler, SoundPlayerMessaged, SoundPlayer
{
	public static SoundPlayerSynthetic open()
			throws SoundPlayerException
	{
		int handle;
		if((handle = (int) MalikSystem.syscall(0L, 0x0038)) == 0)
		{
			throw new SoundPlayerException("SoundPlayerSynthetic.open: " +
					"системе не хватает памяти на создание нового MIDI-проигрывателя или " +
					"система не поддерживает стандарт MIDI.");
		}
		return new SoundPlayerSynthetic(handle);
	}

	private int handle;
	private DataDescriptor descriptor;
	private SoundPlayerListener listener;
	private Object lock;

	private SoundPlayerSynthetic(int handle)
	{
		this.handle = handle;
		this.descriptor = new DataDescriptor();
		this.lock = new Object();
		Run.getInstance().setPlayerMIDIEndTrackHandler(handle, this);
	}

	public void endMIDITrack(int playerMIDIHandle)
	{
		SoundPlayerListener listener;
		if((listener = this.listener) != null && playerMIDIHandle == this.handle)
		{
			listener.loadBlock(this, -1);
		}
	}

	public void sendMessage(int synthesizerMessage)
	{
		MalikSystem.syscall((long) synthesizerMessage, 0x0039);
	}

	public void setSoundPlayerListener(SoundPlayerListener listener)
	{
		this.listener = listener;
	}

	public void close()
	{
		int h;
		synchronized(lock)
		{
			if((h = handle) != 0)
			{
				handle = 0;
				listener = null;
				MalikSystem.syscall((long) h, 0x003f);
				Run.getInstance().setPlayerMIDIEndTrackHandler(h, null);
			}
		}
	}

	public int loadBlock(long[] synthesizerMessages, int offset, int length)
	{
		int h;
		int lim;
		int len;
		int result;
		DataDescriptor d;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		if(synthesizerMessages == null)
		{
			throw new NullPointerException("SoundPlayerSynthetic.loadBlock: " +
					"параметр synthesizerMessages равен нулевой ссылке.");
		}
		if((lim = offset + length) > (len = synthesizerMessages.length) ||
				lim < offset || offset > len || offset < 0)
		{
			throw new ArrayIndexOutOfBoundsException("SoundPlayerSynthetic.loadBlock: " +
					"индекс выходит из диапазона.");
		}
		if(length <= 0)
		{
			return 0;
		}
		synchronized(lock)
		{
			(d = descriptor).setDataInfo(Array.getFirstElementAddress(synthesizerMessages) +
					(offset << 3), length);
			result = (int) MalikSystem.syscall(h, d.getDescriptorAddress(), 0x0039);
		}
		return result;
	}

	public int reset()
	{
		int h;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		return (int) MalikSystem.syscall(h, 0, 0x003a);
	}

	public int start()
	{
		int h;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		return (int) MalikSystem.syscall(h, 1, 0x003a);
	}

	public int stop()
	{
		int h;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		return (int) MalikSystem.syscall(h, 2, 0x003a);
	}

	public int setVolume(int volume)
	{
		int h;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		return (int) MalikSystem.syscall(h,
				volume < 0 ? 0 : (volume > 0x7f ? 0x7f : volume), 0x003e);
	}

	public int setCurrentBlockPosition(int position)
	{
		int h;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		return (int) MalikSystem.syscall(h, position, 0x003c);
	}

	public int getState()
	{
		int h;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		return (int) MalikSystem.syscall(h, 0, 0x003b);
	}

	public int getVolume()
	{
		int h;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		return (int) MalikSystem.syscall(h, -1, 0x003e);
	}

	public int getCurrentBlockPosition()
	{
		int h;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		return (int) MalikSystem.syscall(h, 0, 0x003d);
	}

	public int getCurrentBlockIndex()
	{
		int h;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		return (int) (MalikSystem.syscall(h, 0, 0x003d) >> 32);
	}

	public long getCurrentBlockAndPosition()
	{
		int h;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		return MalikSystem.syscall(h, 0, 0x003d);
	}
}
