/*
    Реализация спецификаций CLDC версии 1.1 (JSR-139), MIDP версии 2.1 (JSR-118)
    и других спецификаций для функционирования компактных приложений на языке
    Java (мидлетов) в среде программного обеспечения Малик Эмулятор.

    Copyright © 2016–2017, 2019–2022 Малик Разработчик

    Это свободная программа: вы можете перераспространять ее и/или изменять
    ее на условиях Меньшей Стандартной общественной лицензии GNU в том виде,
    в каком она была опубликована Фондом свободного программного обеспечения;
    либо версии 3 лицензии, либо (по вашему выбору) любой более поздней версии.

    Эта программа распространяется в надежде, что она будет полезной,
    но БЕЗО ВСЯКИХ ГАРАНТИЙ; даже без неявной гарантии ТОВАРНОГО ВИДА
    или ПРИГОДНОСТИ ДЛЯ ОПРЕДЕЛЕННЫХ ЦЕЛЕЙ. Подробнее см. в Меньшей Стандартной
    общественной лицензии GNU.

    Вы должны были получить копию Меньшей Стандартной общественной лицензии GNU
    вместе с этой программой. Если это не так, см.
    <https://www.gnu.org/licenses/>.
*/

package java.io;

public class DataOutputStream extends OutputStream implements DataOutput
{
    private static int writeUTF(DataOutput stream, String utfData) throws IOException {
        int length;
        int strlen;
        char[] c;
        byte[] result;
        if(utfData == null)
        {
            throw new NullPointerException("DataOutputStream.writeUTF: аргумент utfData равен нулевой ссылке.");
        }
        length = 0;
        utfData.getChars(0, strlen = utfData.length(), c = new char[strlen], 0);
        for(int i = strlen; i-- > 0; )
        {
            int ci;
            length += (ci = c[i]) >= 0x0001 && ci < 0x0080 ? 1 : ci < 0x0800 ? 2 : 3;
        }
        if(length >= 0x00010000)
        {
            throw new UTFDataFormatException("DataOutputStream.writeUTF: размер данных, закодированных кодировкой UTF-8, не может превышать 65535 байт.");
        }
        result = new byte[length];
        length = 0;
        for(int i = 0; i < strlen; i++)
        {
            int ci;
            if((ci = c[i]) >= 0x0001 && ci < 0x0080)
            {
                result[length++] = (byte) ci;
                continue;
            }
            if(ci < 0x0800)
            {
                result[length++] = (byte) (0xc0 | ((ci >> 6) & 0x1f));
                result[length++] = (byte) (0x80 | (ci & 0x3f));
                continue;
            }
            result[length++] = (byte) (0xe0 | ((ci >> 12) & 0x0f));
            result[length++] = (byte) (0x80 | ((ci >> 6) & 0x3f));
            result[length++] = (byte) (0x80 | (ci & 0x3f));
        }
        stream.writeShort(length);
        stream.write(result, 0, length);
        return length + 2;
    }

    protected OutputStream out;

    public DataOutputStream(OutputStream stream) {
        this.out = stream;
    }

    public void close() throws IOException {
        out.close();
    }

    public void flush() throws IOException {
        out.flush();
    }

    public void write(int byteData) throws IOException {
        out.write(byteData);
    }

    public void write(byte[] src, int offset, int length) throws IOException {
        out.write(src, offset, length);
    }

    public final void writeBoolean(boolean booleanData) throws IOException {
        out.write(booleanData ? 1 : 0);
    }

    public final void writeChar(int charData) throws IOException {
        OutputStream stream = out;
        stream.write(charData >> 8);
        stream.write(charData);
    }

    public final void writeFloat(float floatData) throws IOException {
        writeInt(Float.floatToIntBits(floatData));
    }

    public final void writeDouble(double doubleData) throws IOException {
        writeLong(Double.doubleToLongBits(doubleData));
    }

    public final void writeByte(int byteData) throws IOException {
        out.write(byteData);
    }

    public final void writeShort(int shortData) throws IOException {
        OutputStream stream = out;
        stream.write(shortData >> 8);
        stream.write(shortData);
    }

    public final void writeInt(int intData) throws IOException {
        OutputStream stream = out;
        stream.write(intData >> 0x18);
        stream.write(intData >> 0x10);
        stream.write(intData >> 0x08);
        stream.write(intData);
    }

    public final void writeLong(long longData) throws IOException {
        OutputStream stream = out;
        stream.write((int) (longData >> 0x38));
        stream.write((int) (longData >> 0x30));
        stream.write((int) (longData >> 0x28));
        stream.write((int) (longData >> 0x20));
        stream.write((int) (longData >> 0x18));
        stream.write((int) (longData >> 0x10));
        stream.write((int) (longData >> 0x08));
        stream.write((int) longData);
    }

    public final void writeChars(String charsData) throws IOException {
        char[] c;
        OutputStream stream;
        if(charsData == null)
        {
            throw new NullPointerException("DataOutputStream.writeChars: аргумент charsData равен нулевой ссылке.");
        }
        stream = out;
        for(int len = (c = charsData.toCharArray()).length, i = 0; i < len; i++)
        {
            int charData = c[i];
            stream.write(charData >> 8);
            stream.write(charData);
        }
    }

    public final void writeUTF(String utfData) throws IOException {
        writeUTF(this, utfData);
    }
}
