/*
    Реализация спецификаций CLDC версии 1.1 (JSR-139), MIDP версии 2.1 (JSR-118)
    и других спецификаций для функционирования компактных приложений на языке
    Java (мидлетов) в среде программного обеспечения Малик Эмулятор.

    Copyright © 2016–2017, 2019–2022 Малик Разработчик

    Это свободная программа: вы можете перераспространять ее и/или изменять
    ее на условиях Меньшей Стандартной общественной лицензии GNU в том виде,
    в каком она была опубликована Фондом свободного программного обеспечения;
    либо версии 3 лицензии, либо (по вашему выбору) любой более поздней версии.

    Эта программа распространяется в надежде, что она будет полезной,
    но БЕЗО ВСЯКИХ ГАРАНТИЙ; даже без неявной гарантии ТОВАРНОГО ВИДА
    или ПРИГОДНОСТИ ДЛЯ ОПРЕДЕЛЕННЫХ ЦЕЛЕЙ. Подробнее см. в Меньшей Стандартной
    общественной лицензии GNU.

    Вы должны были получить копию Меньшей Стандартной общественной лицензии GNU
    вместе с этой программой. Если это не так, см.
    <https://www.gnu.org/licenses/>.
*/

package malik.emulator.microedition.system.console;

import malik.emulator.microedition.*;
import malik.emulator.microedition.system.*;
import malik.emulator.util.*;

public class DeviceKeyConsoleCommand extends ConsoleCommand
{
    private static final int[] DEVICE_KEY_CODES;
    private static final String[] DEVICE_KEY_NAMES;

    static {
        DEVICE_KEY_CODES = new int[] {
            DeviceSettings.DEVICE_KEY_1, DeviceSettings.DEVICE_KEY_2, DeviceSettings.DEVICE_KEY_3,
            DeviceSettings.DEVICE_KEY_4, DeviceSettings.DEVICE_KEY_5, DeviceSettings.DEVICE_KEY_6,
            DeviceSettings.DEVICE_KEY_7, DeviceSettings.DEVICE_KEY_8, DeviceSettings.DEVICE_KEY_9,
            DeviceSettings.DEVICE_KEY_STAR, DeviceSettings.DEVICE_KEY_0, DeviceSettings.DEVICE_KEY_POUND,
            DeviceSettings.DEVICE_KEY_UP, DeviceSettings.DEVICE_KEY_DOWN, DeviceSettings.DEVICE_KEY_LEFT,
            DeviceSettings.DEVICE_KEY_RIGHT, DeviceSettings.DEVICE_KEY_SELECT, DeviceSettings.DEVICE_KEY_SOFT1,
            DeviceSettings.DEVICE_KEY_SOFT2, DeviceSettings.DEVICE_KEY_CONSOLE, DeviceSettings.DEVICE_KEY_EXITAPP
        };
        DEVICE_KEY_NAMES = new String[] { "1", "2", "3", "4", "5", "6", "7", "8", "9", "*", "0", "#", "вверх", "вниз", "влево", "вправо", "выбор", "лкв", "пкв", "консоль", "выход" };
    }

    private static int getCompjuterKey(String name) {
        return name == null ? 0 : DeviceSettings.stringToKey(name.toUpperCase());
    }

    private static int getDeviceKeyCode(String name) {
        if(name == null) return 0;
        for(int i = DEVICE_KEY_NAMES.length; i-- > 0; ) if(name.equals(DEVICE_KEY_NAMES[i])) return DEVICE_KEY_CODES[i];
        return 0;
    }

    public DeviceKeyConsoleCommand() {
        super(
            "клавиша",
            "Использование:\n клавиша\n клавиша <клавиша устройства>\n клавиша <клавиша устройства> <клавиша компьютера>\n" +
            "Вариант без аргументов выводит на консоль все клавиши устройства и эквивалентные им клавиши компьютера.\n" +
            "Вариант с одним аргументом выводит на консоль клавишу устройства и эквивалентную ей клавишу компьютера.\n" +
            "Вариант с двумя аргументами задаёт новую эквивалентную клавишу компьютера для заданной клавиши устройства.\n" +
            "Следите за тем, чтобы для разных клавиш устройства были заданы разные клавиши компьютера."
        );
    }

    protected void execute(String[] arguments, Console console) {
        int deviceKey;
        int compjuterKey;
        int alen = arguments.length;
        DeviceSettings proxy = DeviceManager.getInstance().getSettings();
        String argument;
        if(alen <= 0)
        {
            /* Вариант без аргументов выводит на консоль все клавиши устройства и эквивалентные им клавиши компьютера. */
            StringBuilder text = new StringBuilder();
            for(int len = DEVICE_KEY_NAMES.length, i = 0; i < len; i++)
            {
                text.append(DEVICE_KEY_NAMES[i]).append(" – ").append(DeviceSettings.keyToString(proxy.getKeyUsedAs(DEVICE_KEY_CODES[i]))).append('\n');
            }
            console.print(text.toString());
            return;
        }
        if(alen == 1)
        {
            /* Вариант с одним аргументом выводит на консоль клавишу устройства и эквивалентную ей клавишу компьютера. */
            if((deviceKey = getDeviceKeyCode(argument = arguments[0])) == 0)
            {
                console.println("Неизвестная клавиша устройства ".concat(argument));
                return;
            }
            console.println((new StringBuilder()).append(argument).append(" – ").append(DeviceSettings.keyToString(proxy.getKeyUsedAs(deviceKey))).toString());
            return;
        }

        /* Вариант с двумя аргументами задаёт новую эквивалентную клавишу компьютера для заданной клавиши устройства. */
        if((deviceKey = getDeviceKeyCode(argument = arguments[0])) == 0)
        {
            console.println("Неизвестная клавиша устройства ".concat(argument));
            return;
        }
        if((compjuterKey = getCompjuterKey(argument = arguments[1])) == 0)
        {
            console.println("Неизвестная клавиша компьютера ".concat(argument));
            return;
        }
        proxy.setKeyUsedAs(deviceKey, compjuterKey);
    }
}
