/*
	Реализация спецификаций CLDC версии 1.1 (JSR-139), MIDP версии 2.1 (JSR-118)
	и других спецификаций для функционирования компактных приложений на языке
	Java (мидлетов) в среде программного обеспечения Малик Эмулятор.

	Copyright © 2016, 2019 Малик Разработчик

	Это свободная программа: вы можете перераспространять ее и/или изменять
	ее на условиях Меньшей Стандартной общественной лицензии GNU в том виде,
	в каком она была опубликована Фондом свободного программного обеспечения;
	либо версии 3 лицензии, либо (по вашему выбору) любой более поздней версии.

	Эта программа распространяется в надежде, что она будет полезной,
	но БЕЗО ВСЯКИХ ГАРАНТИЙ; даже без неявной гарантии ТОВАРНОГО ВИДА
	или ПРИГОДНОСТИ ДЛЯ ОПРЕДЕЛЕННЫХ ЦЕЛЕЙ. Подробнее см. в Меньшей Стандартной
	общественной лицензии GNU.

	Вы должны были получить копию Меньшей Стандартной общественной лицензии GNU
	вместе с этой программой. Если это не так, см.
	<https://www.gnu.org/licenses/>.
*/


package javax.bluetooth;

public class DiscoveryAgent extends Object
{
	public static final int CACHED = 0;
	public static final int PREKNOWN = 1;
	public static final int LIAC = 0x009e8b00;
	public static final int GIAC = 0x009e8b33;
	public static final int NOT_DISCOVERABLE = 0;
	static final int MIN_DISCOVERABLE = 0x009e8b00;
	static final int MAX_DISCOVERABLE = 0x009e8b3f;


	private boolean discoveryDevicesMode;
	private boolean discoveryServicesMode;
	private RemoteDevice[] emptyRemoteDevicesList;
	private DiscoveryDevicesThread discoveryDevicesThread;
	private DiscoveryServicesThread discoveryServicesThread;
	private Object lock;

	DiscoveryAgent()
	{
		this.emptyRemoteDevicesList = new RemoteDevice[0];
		this.lock = new Object();
	}

	public boolean cancelInquiry(DiscoveryListener listener)
	{
		boolean result;
		DiscoveryDevicesThread discovery;
		if(listener == null)
		{
			throw new NullPointerException("DiscoveryAgent.cancelInquiry: " +
					"параметр listener равен нулевой ссылке.");
		}
		synchronized(lock)
		{
			discovery = discoveryDevicesThread;
			if(result = discoveryDevicesMode && discovery != null &&
					discovery.getListener() == listener)
			{
				discovery.interrupt();
			}
		}
		return result;
	}

	public boolean cancelServiceSearch(int transactionID)
	{
		boolean result;
		DiscoveryServicesThread discovery;
		synchronized(lock)
		{
			discovery = discoveryServicesThread;
			if(result = discoveryServicesMode && discovery != null &&
					discovery.getTransactionID() == transactionID)
			{
				discovery.interrupt();
			}
		}
		return result;
	}

	public boolean startInquiry(int accessCode, DiscoveryListener listener)
			throws BluetoothStateException
	{
		int error;
		DiscoveryDevicesThread discovery;
		if(accessCode < MIN_DISCOVERABLE || accessCode > MAX_DISCOVERABLE)
		{
			throw new IllegalArgumentException("DiscoveryAgent.startInquiry: " +
					"недопустимое значение параметра accessCode.");
		}
		if(listener == null)
		{
			throw new NullPointerException("DiscoveryAgent.startInquiry: " +
					"параметр listener равен нулевой ссылке.");
		}
		error = 0;
		synchronized(lock)
		{
			label0:
			{
				if(discoveryDevicesMode)
				{
					error = 1;
					break label0;
				}
				discoveryDevicesMode = true;
				discoveryDevicesThread = discovery = new DiscoveryDevicesThread(
						this, listener);
				discovery.start();
			}
		}
		if(error == 1)
		{
			throw new BluetoothStateException("DiscoveryAgent.startInquiry: " +
					"поиск bluetooth-устройств уже выполняется.");
		}
		return true;
	}

	public int searchServices(int[] attributes, UUID[] uuids,
			RemoteDevice deviceInfo, DiscoveryListener listener)
			throws BluetoothStateException
	{
		int i;
		int j;
		int error;
		int result;
		int attrTemp;
		int attrsLen;
		int uuidsLen;
		int[] attrsCopy;
		UUID[] uuidsCopy;
		UUID uuidTemp;
		DiscoveryServicesThread discovery;
		if(uuids == null)
		{
			throw new NullPointerException("DiscoveryAgent.searchServices: " +
					"параметр uuids равен нулевой ссылке.");
		}
		if(deviceInfo == null)
		{
			throw new NullPointerException("DiscoveryAgent.searchServices: " +
					"параметр deviceInfo равен нулевой ссылке.");
		}
		if(listener == null)
		{
			throw new NullPointerException("DiscoveryAgent.searchServices: " +
					"параметр listener равен нулевой ссылке.");
		}
		if(attributes == null)
		{
			attrsLen = -1;
		}
		else if((attrsLen = attributes.length) <= 0)
		{
			throw new IllegalArgumentException("DiscoveryAgent.searchServices: " +
					"параметр attributes имеет нулевую длину.");
		}
		if((uuidsLen = uuids.length) <= 0)
		{
			throw new IllegalArgumentException("DiscoveryAgent.searchServices: " +
					"параметр uuids имеет нулевую длину.");
		}
		if(attrsLen > 0)
		{
			Array.copy(attributes, 0, attrsCopy = new int[attrsLen], 0, attrsLen);
		} else
		{
			attrsCopy = null;
		}
		for(j = attrsLen; j-- > 1; )
		{
			for(attrTemp = attrsCopy[j], i = j; i-- > 0; )
			{
				if(attrsCopy[i] == attrTemp)
				{
					throw new IllegalArgumentException("DiscoveryAgent.searchServices: " +
							"параметр attributes содержит дубликаты.");
				}
			}
		}
		Array.copy(uuids, 0, uuidsCopy = new UUID[uuidsLen], 0, uuidsLen);
		for(j = 0; j < uuidsLen; j++)
		{
			if((uuidTemp = uuidsCopy[j]) == null)
			{
				throw new NullPointerException("DiscoveryAgent.searchServices: " +
						"параметр uuids содержит нулевые ссылки.");
			}
			for(i = j; i-- > 0; )
			{
				if(uuidTemp.isRealEquals(uuidsCopy[i]))
				{
					throw new IllegalArgumentException("DiscoveryAgent.searchServices: " +
							"параметр uuids содержит дубликаты.");
				}
			}
		}
		error = 0;
		synchronized(lock)
		{
			label0:
			{
				if(discoveryServicesMode)
				{
					error = 1;
					result = 0;
					break label0;
				}
				discoveryServicesMode = true;
				discoveryServicesThread = discovery = new DiscoveryServicesThread(
						this, listener, result = 1);
				discovery.start();
			}
		}
		if(error == 1)
		{
			throw new BluetoothStateException("DiscoveryAgent.searchServices: " +
					"поиск bluetooth-сервисов уже выполняется.");
		}
		return result;
	}

	public RemoteDevice[] retrieveDevices(int option)
	{
		if(option != CACHED && option != PREKNOWN)
		{
			throw new IllegalArgumentException("DiscoveryAgent.retrieveDevices: " +
					"недопустимое значение параметра option.");
		}
		return emptyRemoteDevicesList;
	}

	public String selectService(UUID uuid, int security, boolean master)
			throws BluetoothStateException
	{
		if(uuid == null)
		{
			throw new NullPointerException("DiscoveryAgent.selectService: " +
					"параметр uuid равен нулевой ссылке.");
		}
		if(security != ServiceRecord.NOAUTHENTICATE_NOENCRYPT &&
				security != ServiceRecord.AUTHENTICATE_NOENCRYPT &&
				security != ServiceRecord.AUTHENTICATE_ENCRYPT)
		{
			throw new IllegalArgumentException("DiscoveryAgent.selectService: " +
					"недопустимое значение параметра security.");
		}
		throw new BluetoothStateException("DiscoveryAgent.selectService: " +
				"система не может начать запрос.");
	}

	final void endDevicesDiscovery()
	{
		synchronized(lock)
		{
			discoveryDevicesMode = false;
			discoveryDevicesThread = null;
		}
	}

	final void endServicesDiscovery()
	{
		synchronized(lock)
		{
			discoveryServicesMode = false;
			discoveryServicesThread = null;
		}
	}
}
