/*
    Реализация спецификаций CLDC версии 1.1 (JSR-139), MIDP версии 2.1 (JSR-118)
    и других спецификаций для функционирования компактных приложений на языке
    Java (мидлетов) в среде программного обеспечения Малик Эмулятор.

    Copyright © 2016–2017, 2019–2022 Малик Разработчик

    Это свободная программа: вы можете перераспространять ее и/или изменять
    ее на условиях Меньшей Стандартной общественной лицензии GNU в том виде,
    в каком она была опубликована Фондом свободного программного обеспечения;
    либо версии 3 лицензии, либо (по вашему выбору) любой более поздней версии.

    Эта программа распространяется в надежде, что она будет полезной,
    но БЕЗО ВСЯКИХ ГАРАНТИЙ; даже без неявной гарантии ТОВАРНОГО ВИДА
    или ПРИГОДНОСТИ ДЛЯ ОПРЕДЕЛЕННЫХ ЦЕЛЕЙ. Подробнее см. в Меньшей Стандартной
    общественной лицензии GNU.

    Вы должны были получить копию Меньшей Стандартной общественной лицензии GNU
    вместе с этой программой. Если это не так, см.
    <https://www.gnu.org/licenses/>.
*/

package malik.emulator.media.sound;

import malik.emulator.application.*;
import malik.emulator.util.*;

public final class SyntheticSoundPlayer extends SoundPlayer
{
    public static SyntheticSoundPlayer open() throws SoundPlayerException {
        int handle;
        if((handle = (int) MalikSystem.syscall(0L, 0x0038)) == 0)
        {
            throw new SoundPlayerException(
                "SyntheticSoundPlayer.open: системе не хватает памяти на создание нового проигрывателя или возможности системы не поддерживают цифровой интерфейс музыкальных инструментов."
            );
        }
        return new SyntheticSoundPlayer(handle);
    }

    int handle;
    private final DataDescriptor descriptor;

    private SyntheticSoundPlayer(int handle) {
        this.handle = handle;
        this.descriptor = new DataDescriptor();
        Run.instance.setMIDIPlayerListener(handle, new MIDIPlayerListener() {
            public void endOfTrack(int playerHandle) {
                SyntheticSoundPlayer parent;
                SoundPlayerListener listener;
                if(playerHandle == (parent = SyntheticSoundPlayer.this).handle && (listener = parent.listener) != null) listener.endOfBlock(parent, -1);
            }
        });
    }

    public void close() {
        synchronized(descriptor)
        {
            int h;
            if((h = handle) != 0)
            {
                handle = 0;
                listener = null;
                MalikSystem.syscall((long) h, 0x003f);
                Run.instance.setMIDIPlayerListener(h, null);
            }
        }
    }

    public int loadBlock(long[] midiMessages, int offset, int length) throws SoundPlayerException {
        int error;
        int result;
        DataDescriptor d;
        if(midiMessages == null)
        {
            throw new NullPointerException("SyntheticSoundPlayer.loadBlock: аргумент midiMessages равен нулевой ссылке.");
        }
        Array.checkBound("SyntheticSoundPlayer.loadBlock", midiMessages.length, offset, length);
        if(length <= 0) return 0;
        error = 0;
        synchronized(d = descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    result = 0;
                    break label0;
                }
                d.setDataInfo(Array.getFirstElementAddress(midiMessages) + (offset << 3), length);
                result = (int) MalikSystem.syscall(h, d.getDescriptorAddress(), 0x0039);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SyntheticSoundPlayer.loadBlock: проигрыватель закрыт.");
        }
        return result;
    }

    public int reset() throws SoundPlayerException {
        int result;
        int error = 0;
        synchronized(descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    result = 0;
                    break label0;
                }
                result = (int) MalikSystem.syscall(h, 0, 0x003a);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SyntheticSoundPlayer.reset: проигрыватель закрыт.");
        }
        return result;
    }

    public int start() throws SoundPlayerException {
        int result;
        int error = 0;
        synchronized(descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    result = 0;
                    break label0;
                }
                result = (int) MalikSystem.syscall(h, 1, 0x003a);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SyntheticSoundPlayer.start: проигрыватель закрыт.");
        }
        return result;
    }

    public int stop() throws SoundPlayerException {
        int result;
        int error = 0;
        synchronized(descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    result = 0;
                    break label0;
                }
                result = (int) MalikSystem.syscall(h, 2, 0x003a);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SyntheticSoundPlayer.stop: проигрыватель закрыт.");
        }
        return result;
    }

    public int setVolume(int volume) throws SoundPlayerException {
        int result;
        int error = 0;
        synchronized(descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    result = 0;
                    break label0;
                }
                result = (int) MalikSystem.syscall(h, super.setVolume(volume), 0x003e);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SyntheticSoundPlayer.setVolume: проигрыватель закрыт.");
        }
        return result;
    }

    public int setCurrentBlockPosition(int position) throws SoundPlayerException {
        int result;
        int error = 0;
        synchronized(descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    result = 0;
                    break label0;
                }
                result = (int) MalikSystem.syscall(h, position, 0x003c);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SyntheticSoundPlayer.setCurrentBlockPosition: проигрыватель закрыт.");
        }
        return result;
    }

    public int getState() throws SoundPlayerException {
        int result;
        int error = 0;
        synchronized(descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    result = 0;
                    break label0;
                }
                result = (int) MalikSystem.syscall((long) h, 0x003b);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SyntheticSoundPlayer.getState: проигрыватель закрыт.");
        }
        return result;
    }

    public int getVolume() throws SoundPlayerException {
        int result;
        int error = 0;
        synchronized(descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    result = 0;
                    break label0;
                }
                result = (int) MalikSystem.syscall(h, -1, 0x003e);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SyntheticSoundPlayer.getVolume: проигрыватель закрыт.");
        }
        return result;
    }

    public int getCurrentBlockIndex() throws SoundPlayerException {
        if(handle == 0)
        {
            throw new SoundPlayerException("SyntheticSoundPlayer.getCurrentBlockIndex: проигрыватель закрыт.");
        }
        return 0;
    }

    public int getCurrentBlockPosition() throws SoundPlayerException {
        int result;
        int error = 0;
        synchronized(descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    result = 0;
                    break label0;
                }
                result = (int) MalikSystem.syscall(h, 0, 0x003d);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SyntheticSoundPlayer.getCurrentBlockPosition: проигрыватель закрыт.");
        }
        return result;
    }

    public void sendMessage(int midiMessage) throws SoundPlayerException {
        int error = 0;
        DataDescriptor d;
        synchronized(d = descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    break label0;
                }
                d.setDataInfo(midiMessage & 0x00ffffff, 0);
                MalikSystem.syscall(h, d.getDescriptorAddress(), 0x0039);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SyntheticSoundPlayer.sendMessage: проигрыватель закрыт.");
        }
    }
}
