package com.mobilebasic;

/*
 * Mobile BASIC 1.9 Copyright (c) 2003-2010, David Firth
 *
 * This software is released as Open Source and you are free to modify it
 * and build derived versions for your own use. If you have made any changes
 * that you would like included in the main version then they should be
 * sent to the address below.
 *
 * Patches: E-mail to david@mobilebasic.com
 * Website: http://www.mobilebasic.com/
 */

public class Float
{
  public static final int ZERO = 0x00000000;      // fromString("0.0")

  static final int E = 0x14bd2500;         // fromString("2.718282")
  static final int PI = 0x17f7ec80;        // fromString("3.141593")
  static final int PIOVER2 = 0xbfbf600;    // fromString("1.570796")
  static final int PIOVER4 = 0x3bebcf7f;   // fromString("0.7853982")
  static final int TWOPI = 0x2fefd880;     // fromString("6.283185")
  static final int ONE = 0x07a12000;       // fromString("1.0")
  static final int MINUS_ONE = 0xf85ee000; // fromString("-1.0")
  static final int TWO = 0x0f424000;       // fromString("2.0")
  //static int MINUS_NINE = 0xbb55e000;
  
  // DEGRAD = (2. * PI) / 360.0 = 0.017453292519943295769236907684886
  static final int DEGRAD = 0x0d50d8fe; // Float.MakeFP(1745329, -2, false)

  // RADDEG = 360.0 / (2.0 * PI) = 57.295779513082320876798154814105
  static final int RADDEG = 0x2bb69501; // Float.MakeFP(5729578, 1, false)

  static final int LIMIT = 1000; // Convergence limit
  
  static public int MakeFP(long val, int exp, boolean adjustExp)
  {
    while (Math.abs(val) > 9999999)
    {
      val = val / 10;
      if (adjustExp)
          exp += 1;
    }

    /*
     * If digits is zero then it is not possible to multiply it
     * by 10 any number of times such that the answer will be
     * greater than 1000000. It will always be zero!
     * The result must also be zero since 0 * x = 0
     */

    if (Math.abs(val) > 0)
    {
      while (Math.abs(val) < 1000000)
      {
        val = val * 10;
        if (adjustExp)
            exp -= 1;
      }
    }
    else
    {
      exp = 0;
      val = 0;
    }

    /*
     * Create result Float
     */

    int result;

    if (exp < 0)
      exp = exp + 128;

    result = ((int)val << 7) | exp;

    return result;
  }

  static public int itof(int ival)
  {
    return MakeFP(ival, 6, true);
  }

  static public int ftoi(int fp)
  {
    long val = fp >> 7;
    int exp = fp & 0x7f;

    if (exp > 0x3f)
      exp = exp - 128;

    //System.out.println("ftoi(" + toString(fp) + ")");
    //System.out.println("  exp=" + exp);
    //System.out.println("  val=" + val);

    // if (exp >= 0)
    //{
      int shift = 6 - exp;

      while (shift > 0)
      {
        val = val / 10;
        shift--;
      }

      while (shift < 0)
      {
        val = val * 10;
        shift++;
      }
    //}

    //System.out.println("  val=" + val);

    return (int)val;
  }

  static public String toString(int fp)
  {
    StringBuffer sb = new StringBuffer();

    int val = fp >> 7;
    int exp = fp & 0x7f;
    if (exp > 0x3f)
      exp = exp - 128;

    //sb.append("sign=" + sign + ", exp=" + exp + ", val=" + val + " ");

    if (val < 0)
    {
      sb.append('-');
      val = -val;
    }

    int divisor = 1000000;

    if ((exp > 6) || (exp < -6))
    {
      boolean pointFlag = false;

      while (divisor > 0)
      {
        int digit = val / divisor;
        val = val - digit * divisor;

        sb.append((char)(0x30 + digit));

        divisor = divisor / 10;

        if (!pointFlag)
        {
          sb.append('.');
          pointFlag = true;
        }
      } 

      sb.append("E");
      if (exp>=0)
        sb.append('+');
      sb.append(exp);
    }
    else if (exp >= 0)
    {
      while (divisor > 0)
      {
        int digit = val / divisor;
        val = val - digit * divisor;

        sb.append((char)(0x30 + digit));

        divisor = divisor / 10;

        if ((exp == 0) && (divisor > 0))
          sb.append(".");

        exp = exp - 1;
        if ((exp < -1) && (val == 0))
            break;
      } 
    }
    else	//  exp = -1 or less
    {
      sb.append("0.");

      while (divisor > 0)
      {
        exp = exp + 1;

        if (exp < 0)
        {
          sb.append('0');
        }
        else
        {
          int digit = val / divisor;
          val = val - digit * divisor;

          sb.append((char)(0x30 + digit));

          divisor = divisor / 10;
        }

        if (val == 0)
            break;
      }
    }
   
    return sb.toString(); 
  }

  static public int fromString(String ascii)
  {
    int sign = 1;
    int exp = -1;
    int val = 0;
    int expVal = 0;
    int expSign = 1;
    int fp;

    final int ERROR = -1;
    final int SIGN_OR_DIGIT = 0;
    final int DIGIT1 = 1;
    final int DIGIT2 = 2;
    final int EXPSIGN = 3;
    final int GETEXP = 4;

    int state = SIGN_OR_DIGIT;
 
    int len = ascii.length();

    int offset = 0;
    while (offset < len)
    {
      char ch = ascii.charAt(offset);

      switch (state)
      {
        case ERROR :
          throw new FloatException(FloatException.INVALID_FLOATING_POINT_STRING);

        case SIGN_OR_DIGIT :
          if (ch == '-')
          {
            offset++;
            sign = -1;
            state = DIGIT1;
          }
          else if ((ch >= '0') && (ch <= '9'))
          {
            state = DIGIT1;
          }
          else
          {
            state = ERROR;
          }
          break;

        case DIGIT1 :
          if ((ch >= '0') && (ch <= '9'))
          {
            val = val * 10 + (int)(ch - '0');
            if (val != 0)
              exp++;
            offset++;
          }
          else if (ch == '.')
          {
            state = DIGIT2;
            offset++;
          }
          else if ((ch == 'E') || (ch == 'e'))
          {
            state = EXPSIGN;
            offset++;
          }
          else
          {
            state = ERROR;
          }
          break;

        case DIGIT2 :
          if ((ch >= '0') && (ch <= '9'))
          {
            val = val * 10 + (int)(ch - '0');
            if (val == 0)
              exp--;
            offset++;
          }
          else if ((ch == 'E') || (ch == 'e'))
          {
            state = EXPSIGN;
            offset++;
          }
          else
          {
            state = ERROR;
          }
          break;

        case EXPSIGN :
          if (ch == '+')
          {
            expSign = 1;
            offset++;
            state = GETEXP;
          }
          else if (ch == '-')
          {
            expSign = -1;
            offset++;
            state = GETEXP;
          }
          else
          {
            state = ERROR;
          }
          break;

        case GETEXP :
          if ((ch >= '0') && (ch <= '9'))
          {
            expVal = expVal * 10 + (int)(ch - '0');
            offset++;
          }
          else
          {
            state = ERROR;
          }
          break;

        default :
          state = ERROR;
          break;
      }
    }

    fp = MakeFP(val*sign, exp + expVal * expSign, false);

    return fp;
  }

  static public int Add(int fp1, int fp2)
  {
    long val1 = fp1 >> 7;
    long val2 = fp2 >> 7;
    int exp1 = fp1 & 0x7f;
    int exp2 = fp2 & 0x7f;

    if (exp1 > 0x3f)
      exp1 = exp1 - 128;

    if (exp2 > 0x3f)
      exp2 = exp2 - 128;

    /*
     * Align values so that they can be added
     */
/*
    if (exp1 > exp2)
    {
        int expDiff = exp1 - exp2; // 1
        for (int i=0;i<expDiff;i++)
        {
            val1 = val1 * 10;
            exp1--;
        }
    }
    else if (exp1 < exp2)
    {
        int expDiff = exp2 - exp1;
        for (int i=0;i<expDiff;i++)
        {
            val2 = val2 * 10;
            exp2--;
        }
    }
*/
    while (exp1 > exp2)
    {
      val2 = val2 / 10;
      exp2++;
    }

    while (exp1 < exp2)
    {
      val1 = val1 / 10;
      exp1++;
    }

    val1 = val1 + val2;

    return MakeFP(val1, exp1, true);
  }

  static public int Subtract(int fp1, int fp2)
  {
//System.out.print(toString(fp1) + "-" + toString(fp2) + "=");
    long val1 = fp1 >> 7;
    long val2 = fp2 >> 7;
    int exp1 = fp1 & 0x7f;
    int exp2 = fp2 & 0x7f;

    if (exp1 > 0x3f)
      exp1 = exp1 - 128;

    if (exp2 > 0x3f)
      exp2 = exp2 - 128;

    /*
     * Align values so that they can be subtracted
     */
/*
    if (exp1 > exp2)
    {
      int expDiff = exp1 - exp2; // 1
      for (int i=0;i<expDiff;i++)
      {
        val1 = val1 * 10;
        exp1--;
      }
    }
    else if (exp1 < exp2)
    {
      int expDiff = exp2 - exp1;
      for (int i=0;i<expDiff;i++)
      {
        val2 = val2 * 10;
        exp2--;
      }
    }
*/

    while (exp1 > exp2)
    {
      val2 = val2 / 10;
      exp2++;
    }

    while (exp1 < exp2)
    {
      val1 = val1 / 10;
      exp1++;
    }

    val1 = val1 - val2;

    return MakeFP(val1, exp1, true);
  }

  static public int Multiply(int fp1, int fp2)
  {
    long val1 = fp1 >> 7;
    long val2 = fp2 >> 7;
    int exp1 = fp1 & 0x7f;
    int exp2 = fp2 & 0x7f;

    if (exp1 > 0x3f)
      exp1 = exp1 - 128;

    if (exp2 > 0x3f)
      exp2 = exp2 - 128;

    val1 = (val1 * val2) / 1000000;
    exp1 += exp2;

    return MakeFP(val1, exp1, true);
  }

  static public int Divide(int fp1, int fp2)
  {
    long val1 = fp1 >> 7;
    long val2 = fp2 >> 7;
    int exp1 = fp1 & 0x7f;
    int exp2 = fp2 & 0x7f;

    if (exp1 > 0x3f)
      exp1 = exp1 - 128;

    if (exp2 > 0x3f)
      exp2 = exp2 - 128;

    val1 = (val1 * 1000000) / val2;
    exp1 -= exp2;

    return MakeFP(val1, exp1, true);
  }

  /*
   *
   * fp1 < fp2 = -1
   * fp1 = fp2 =  0
   * fp1 > fp2 =  1
   */

  static public int Compare(int fp1, int fp2)
  {
    int result;

    if (fp1 == fp2)
    {
      result = 0;
    }
    else
    {
      int diff = Subtract(fp1, fp2);
//System.out.println(toString(fp1) + " - " + toString(fp2) + " = " + toString(diff));
      if (diff < 0)
        result = -1;
      else
        result = 1;
    }

    return result;
  }

  static public int Negate(int fp)
  {
    long val = fp >> 7;
    int exp = fp & 0x7f;

    if (exp > 0x3f)
      exp = exp - 128;

    val = -val;

    return MakeFP(val, exp, true);
  }


  /*
   * Log(x+1) = x - (x^2 / 2) + (x^3 / 3) - (x^4 / 4) + ...
   *
   * where 1 > X >= 0
   */

  static public int log(int x)
  {
    //System.out.println("log(" + toString(x) + ") entered");

    int result = ZERO;

    if (Compare(x, ZERO) > 0)
    {
      int sum = ZERO;

      while (Compare(x, ONE) < 1)
      {
        x = Multiply(x, E);
        sum = Subtract(sum, ONE);
//System.out.println("log1> sum=" + toString(sum) + " x=" + toString(x));
      }

      while (Compare(x, ONE) >= 1)
      {
        x = Divide(x, E);
        sum = Add(sum, ONE);
//System.out.println("log2> sum=" + toString(sum) + " x=" + toString(x));
      }

//System.out.println("log> sum" + toString(sum));

      x = Subtract(x, ONE);

      int xPower = Negate(Multiply(x, x));

      int div = TWO;

      result = x;
      int lastResult = x + 1;

      int count = 0;
      
      while (result != lastResult)
      {
        lastResult = result;
        result = Add(result,Divide(xPower, div));
//System.out.println("log> result" + toString(result));
        xPower = Negate(Multiply(xPower, x));
        div = Add(div, ONE);

        if (count++ > LIMIT)
            break;
      }

      // System.out.println("log(" + toString(x) + ") finished");

      result = Add(sum, result);
    }
    else
    {
      throw new FloatException(FloatException.INVALID_VALUE);
    }

    // System.out.println("result=" + toString(result));
    return result;
  }

  /*
   * Exp(x) = 1 + x + (x^2 / 2!) + (x^3 / 3!) + (x^4 / 4!) + ...
   * This function doesn't converge for small -ve values of 
   * < about -6. This seems to be due to integer rounding errors.
   *
   * This can be solved by remembering that: e^(-x) = 1/(e^x)
   */

  static public int exp(int x)
  {
    int result;

    if (Compare(x, ZERO) < 0)
    {
      x = Negate(x);
      result = exp2(x);
      result = Divide(ONE, result);
    }
    else
    {
      result = exp2(x);
    }

    return result;
  }

  static public int exp2(int x)
  {
    int result = ONE;

    int xPower = x;
    int fact = ONE;
    int n = ONE;

    int lastResult = result + 1;

    int count = 0;
    
    while (result != lastResult)
    {
//System.out.print("> ");
      lastResult = result;
//System.out.print(" xPower=" + toString(xPower));
//System.out.print(" fact=" + toString(fact));
      int xPowerDivFact = Divide(xPower, fact);
//System.out.print(" div=" + toString(xPowerDivFact));
//System.out.print(toString(result) + "+" + toString(xPowerDivFact) + "=");
      result = Add(result, xPowerDivFact);
//System.out.print(" result=" + toString(result));
      xPower = Multiply(xPower, x);
      n = Add(n, ONE);
      fact = Multiply(fact, n);

//System.out.println("");

      if (count++ > LIMIT)
        break;
    }

    if (Compare(result, ZERO) < 0)
      result = ZERO;

    return result;
  }

/*
  static public int exp3(int x)
  {
    int result = ONE;

    if (Compare(x, MINUS_NINE) > 0)
    {
      int xPower = x;
      int fact = ONE;
      int n = ONE;

      int lastResult = result + 1;

      int count = 0;
    
      while (result != lastResult)
      {
//System.out.print("> ");
        lastResult = result;
//System.out.print(" xPower=" + toString(xPower));
//System.out.print(" fact=" + toString(fact));
        int xPowerDivFact = Divide(xPower, fact);
//System.out.print(" div=" + toString(xPowerDivFact));
//System.out.print(toString(result) + "+" + toString(xPowerDivFact) + "=");
        result = Add(result, xPowerDivFact);
//System.out.print(" result=" + toString(result));
        xPower = Multiply(xPower, x);
        n = Add(n, ONE);
        fact = Multiply(fact, n);

//System.out.println("");

        if (count++ > LIMIT)
          break;
      }

      if (Compare(result, ZERO) < 0)
        result = ZERO;
    }
    else
    {
      result = ZERO;
    }

    return result;
  }
*/

  /*
   * Sqrt(x) = Exp(Log(x)/2)
   */

  static public int sqrt(int x)
  {
    int result = Compare(x, ZERO);

    if (result > 0)
    {
      result = exp(Divide(log(x),TWO));
    }
    else if (result == 0)
    {
      result = ZERO;
    }
    else
    {
      throw new FloatException(FloatException.INVALID_VALUE);
    }

    return result;
  }

  /*
   * Power(x,y) = Exp(Log(x) * y)
   */

  static public int pow(int x, int y)
  {
    int result = Compare(x, ZERO);

    if (result > 0)
    {
      result = exp(Multiply(log(x),y));
    }
    else if (result == 0)
    {
      result = ZERO;
    }
    else
    {
      /*
       * We can't do raise negative values of x unless y is integer
       */

      int yi = itof(ftoi(y));
      int yd = Subtract(y, yi);

      // System.out.println("y = " + toString(y));
      // System.out.println("yi = " + toString(yi));
      // System.out.println("yd = " + toString(yd));

      if (Compare(yd, ZERO) == 0)
      {
        x = Negate(x);
        result = exp(Multiply(log(x),y));
        if ((yi & 0x01) == 1)
          result = Negate(result);
      }
      else
      {
        throw new FloatException(FloatException.INVALID_VALUE);
      }
    }
    
    return result;
  }

  /*
   * Sin(x) = x - (x^3 / 3!) + (x^5 / 5!) - (x^7 / 7!) + ...
   *
   * Also
   *   Sin(x) = t/sqrt(1+t^2) where t = tan(x)
   */

  static public int sin(int x)
  {
    int num = itof(ftoi(Divide(x, TWOPI)));
    int sub = Multiply(TWOPI, num);
    x = Subtract(x, sub);

    int result = ZERO;
    int xPower = x;
    int fact = ONE;
    int n = ONE;

    int lastResult = result + 1;

    int count = 0;
    
    while (result != lastResult)
    {
      lastResult = result;
      result = Add(result, Divide(xPower, fact));
      xPower = Negate(Multiply(xPower, Multiply(x,x)));
      n = Add(n, ONE);
      fact = Multiply(fact, n);
      n = Add(n, ONE);
      fact = Multiply(fact, n);

      if (count++ > LIMIT)
          break;
    }

    return result;
  }

  static public int sind(int x)
  {
      return sin(Multiply(x, DEGRAD));
  }
  
  /*
   * Cos(x) = 1 - (x^2 / 2!) + (x^4 / 4!) - (x^6 / 6!) + ...
   *
   * Also
   *   Cos(x) = 1 / sqrt(1+t^2) where t = tan(x)
   *   Cos(x) = Sin(X + Pi/2)
   */

  static public int cos(int x)
  {
    return sin(Add(x,PIOVER2));
/*
    int num = itof(ftoi(Divide(x, TWOPI)));
    int sub = Multiply(TWOPI, num);
    x = Subtract(x, sub);

    int result = ONE;
    int xPower = Negate(Multiply(x,x));
    int fact = TWO;
    int n = TWO;

    int lastResult = result + 1;

    int count = 0;
    
    while (result != lastResult)
    {
      lastResult = result;
      result = Add(result, Divide(xPower, fact));
      xPower = Negate(Multiply(xPower, Multiply(x,x)));
      n = Add(n, ONE);
      fact = Multiply(fact, n);
      n = Add(n, ONE);
      fact = Multiply(fact, n);

      if (count++ > LIMIT)
          break;
    }

    return result;
*/
  }

  static public int cosd(int x)
  {
      return cos(Multiply(x, DEGRAD));
  }
  
  /*
   * Tan(x) = Sin(x) / Cos(x)
   */

  static public int tan(int x)
  {
    int cosX = cos(x);
    int result;

    if (Compare(cosX, ZERO) != 0)
      result = Divide(sin(x), cosX);
    else
      throw new FloatException(FloatException.INVALID_VALUE);

    return result;
  }

  static public int tand(int x)
  {
      return tan(Multiply(x, DEGRAD));
  }
  
  /*
   * ASin(x) = ATan(x / sqrt(1-x*x))
   *
   * Also
   *   ASin(x) = ACot(sqrt(1-x*x)/x)
   */

  static public int asin(int x)
  {
    int result;

    if ((Compare(x, MINUS_ONE) >= 0) && (Compare(x, ONE) <= 0))
    {
      result = Multiply(x,x);
      result = Subtract(ONE,result);
      result = sqrt(result);
      if (Compare(result, ZERO) != 0)
      {
        result = Divide(x, result);
        result = atan(result);
      }
      else
      {
        if (Compare(x, ZERO) < 0)
          result = Negate(PIOVER2);
        else
          result = PIOVER2;
      }
    }
    else
    {
      throw new FloatException(FloatException.INVALID_VALUE);
    }
    
    return result;
  }

  static public int asind(int x)
  {
      return Multiply(asin(x), RADDEG);
  }
  
  /*
   * ACos(x) = ATan(sqrt(1-x*x)/x)
   *
   * Also
   *   ACos(x) = ACot(x/sqrt(1-x*x))
   *   ACos(x) = Asin(sqrt(1-x*x))
   *   ACos(x) = 0.5 * PI * asin(-x)
   *   ACos(x) = 0.5 * PI - asin(x)
   */

  static public int acos(int x)
  {
    return Subtract(Divide(PI,TWO),asin(x));
  }

  static public int acosd(int x)
  {
      return Multiply(acos(x), RADDEG);
  }

  /*
   * ATan(x) = x - (x^3 / 3) + (x^5 / 5) - (x^7 / 7) + ...
   */

  static public int atan(int x)
  {
    int MinusONE = Negate(ONE);
    int PIover2 = MakeFP(1570796, 0, false);		// PI over 2

    int result = 0;

    int cmp1 = Compare(x,MinusONE);
    int cmp2 = Compare(x,ONE);
    if (cmp1 == 0)					// tan2 doesn't converge
    {
      result = Negate(PIOVER4);				// so hardcode result
    }
    else if (cmp2 == 0)					// tan2 doesn't converge
    {
      result = PIOVER4;					// so hardcode result
    }
    else if (cmp1 < 0)
    {
      int oneOverX = Divide(ONE, x);
      result = atan2(oneOverX);
      //System.out.println(toString(x) + "<" + toString(MinusONE) + " result = " + toString(result));
      result = Subtract(Negate(PIover2), result);
    }
    else if (cmp2 > 0)
    {
      int oneOverX = Divide(ONE, x);
      result = atan2(oneOverX);
      //System.out.println(toString(x) + ">" + toString(MinusONE) + " result = " + toString(result));
      result = Subtract(PIover2, result);
    }
    else
    {
      result = atan2(x);
    }

    return result;
  }

  static public int atand(int x)
  {
      return Multiply(atan(x), RADDEG);
  }

  /*
   * atan2 is just a support routine for atan. It is not a atan2(x,y) function!
   */

  static private int atan2(int x)
  {
    //System.out.println("atan2(" + toString(x) + ") starting");
    int result = ZERO;
    int xPower = x;
    int n = ONE;

    int lastResult = result + 1;

    int count = 0;
    
    while (result != lastResult)
    {
      lastResult = result;
      result = Add(result, Divide(xPower, n));
//System.out.println("atan(" + toString(x) + ", n=" + toString(n) + ", result=" + toString(result) + ", lastResult=" + toString(lastResult) + ", xPower=" + toString(xPower));
      xPower = Negate(Multiply(xPower, Multiply(x,x)));
      n = Add(n, TWO);
      if (count++ > LIMIT)
          break;
    }
    //System.out.println("atan2(" + toString(x) + ") finished");
/*

    System.out.println("calculating atan(" + toString(x) + ")");

    int result = ZERO;
    int xPower = x;
    int n = ONE;

    int lastResult = result + 1;

    while (result != lastResult)
    {
      lastResult = result;

      int minus1 = Negate(ONE);
//System.out.println("minus1 = " + toString(minus1));
      int top = pow(minus1, Add(n, ONE));
//System.out.println("top = " + toString(top));
      int twoNMinus1 = Subtract(Multiply(TWO,n), ONE);
//System.out.println("twoNMinus1 = " + toString(twoNMinus1));
      xPower = pow(x, twoNMinus1);
//System.out.println("xPower = " + toString(xPower));
      result = Add(result, Multiply(Divide(top,twoNMinus1),xPower));
      System.out.println("result=" + toString(result));

      n = Add(n, ONE);
    }

System.out.println("atan(" + toString(x) + ")=" + toString(result));
*/
    return result;
  }
/*
  static private void AddTest()
  {
    int fp1 = (9912345 << 7) | 0x01;	//  99.12345
    int fp2 = (1234567 << 7) | 0x00;	//   1.234567
    int fp3 = Add(fp1, fp2);		// 100.358017

    System.out.println(toString(fp1) + " + " + toString(fp2) + " = " + toString(fp3));
  }

  static private void MultiplyTest()
  {
    int fp1 = (-5000000 << 7) | 0x01;	// -50.0
    int fp2 = (5000000 << 7) | 0x7d;	//   0.005
    int fp3 = Multiply(fp1, fp2);		//  -0.25

    System.out.println(toString(fp1) + " * " + toString(fp2) + " = " + toString(fp3));
  }

  static private void DivideTest()
  {
    int fp1 = (3141593 << 7) | 0x00;	// 3.141593
    int fp2 = (5000000 << 7) | 0x06;	// 5000000
    int fp3 = Divide(fp1, fp2);		// 6.283196E-07

    System.out.println(toString(fp1) + " / " + toString(fp2) + " = " + toString(fp3));
  }

  static private void SubtractTest()
  {
    int fp1 = (5000000 << 7) | 0x01;	// 50.0
    int fp2 = (-5100000 << 7) | 0x00;	// -5.1
    int fp3 = Subtract(fp1, fp2);		// 55.1

    System.out.println(toString(fp1) + " - " + toString(fp2) + " = " + toString(fp3));
  }

  static private void CompareTest()
  {
    int res;

    res = Compare(itof(50), itof(40));
    System.out.println("50 CMP 40 = " + res);
    res = Compare(itof(40), itof(40));
    System.out.println("40 CMP 40 = " + res);
    res = Compare(itof(30), itof(40));
    System.out.println("30 CMP 40 = " + res);
  }

  static public void ATanTest(int x)
  {
    //System.out.println("ATanTest: x=" + toString(x));
    int fp2 = tan(x);
    //System.out.println("ATanTest: fp2=" + toString(fp2));
    int fp3 = atan(fp2);
    //System.out.println("ATanTest: fp3=" + toString(fp3));
    System.out.println("tan(" + toString(x) + ")=" + toString(fp2) + ", atan(" + toString(fp2) + ")=" + toString(fp3));
  }

  static public void main (String[] args)
  {
    AddTest();
    MultiplyTest();
    DivideTest();
    SubtractTest();
    CompareTest();

    int fp1;
    int fp2;
    int fp3;

    ATanTest(MakeFP(-1570796, 0, false));		// -90 Degrees (E)
    ATanTest(MakeFP(-1308997, 0, false));		// -75 Degrees
    ATanTest(MakeFP(-1047198, 0, false));		// -60 Degrees
    //ATanTest(MakeFP(-7853982, -1, false));		// -45 Degrees
    ATanTest(MakeFP(-5235988, -1, false));		// -30 Degrees
    ATanTest(MakeFP(-2617994, -1, false));		// -15 Degrees
    ATanTest(ZERO);					//   0 Degrees
    ATanTest(MakeFP(2617994, -1, false));		//  15 Degrees
    ATanTest(MakeFP(5235988, -1, false));		//  30 Degrees
    ATanTest(MakeFP(7853982, -1, false));		//  45 Degrees
    ATanTest(MakeFP(1047198, 0, false));		//  60 Degrees
    ATanTest(MakeFP(1308997, 0, false));		//  75 Degrees
    ATanTest(MakeFP(1553343, 0, false));		//  89 Degrees (E)
    ATanTest(MakeFP(1569051, 0, false));		//  89.9 Degrees (E)
    ATanTest(MakeFP(1570796, 0, false));		//  90 Degrees (E)
    ATanTest(MakeFP(1570796, 0, false));		//  90 Degrees (E)
    ATanTest(MakeFP(2007128, 0, false));		// 115 Degrees (E)

    ATanTest(MakeFP(-2007128, 0, false));		// Test Value

    fp1 = MakeFP(-5000000,0,false);
    fp2 = exp(fp1);
    System.out.println("exp(" + toString(fp1) + ") = " + toString(fp2));

    fp1 = MakeFP(-3000000,0,false);
    fp2 = MakeFP(3000000,0,false);
    fp3 = pow(fp1,fp2);
    System.out.println("pow(" + toString(fp1) + "," + toString(fp2) + ") = " + toString(fp3));

    fp1 = MakeFP(3000000,0,false);
    fp2 = log(fp1);
    System.out.println("log(" + toString(fp1) + ") = " + toString(fp2));

    fp1 = itof(-512);

    System.out.println("itof(-512) = " + toString(fp1));

    fp1 = MakeFP(-3142159, 0, false);
    System.out.println("ftoi(" + toString(fp1) + ") = " + ftoi(fp1));
    System.out.println("Negate(" + toString(fp1) + ") = " + toString(Negate(fp1)));

    fp1 = MakeFP(8100000, 1, false);
    fp2 = log(fp1);
    System.out.println("log(" + toString(fp1) + ") = " + toString(fp2));

    fp1 = exp(fp2);
    System.out.println("exp(" + toString(fp2) + ") = " + toString(fp1));

    fp1 = MakeFP(8100000, 1, false);
    fp2 = sqrt(fp1);
    System.out.println("sqrt(" + toString(fp1) + ") = " + toString(fp2));

    fp1 = pow(fp2, TWO);
    System.out.println("pow(" + toString(fp2) + "," + toString(TWO) + ") = " + toString(fp1));

    fp1 = PI;
    fp2 = sin(fp1);
    System.out.println("sin(" + toString(fp1) + ") = " + toString(fp2));
    fp2 = cos(fp1);
    System.out.println("cos(" + toString(fp1) + ") = " + toString(fp2));

    fp1 = MakeFP(7853982, -1, false);
    fp1 = MakeFP(1570622, 0, false);
    fp2 = tan(fp1);
    System.out.println("tan(" + toString(fp1) + ") = " + toString(fp2));

    fp2 = MakeFP(5000000, 0, false);
    fp1 = Divide(PI,fp2);
    //fp1 = MakeFP(7853981, -1, false);	// 45 Degrees
    //fp1 = MakeFP(5235987, -1, false);	// 30 Degrees

    int sinX = sin(fp1);
    int cosX = cos(fp1);
    int tanX = tan(fp1);

    System.out.println("sin(" + toString(fp1) + ") = " + toString(sinX));
    System.out.println("cos(" + toString(fp1) + ") = " + toString(cosX));
    System.out.println("tan(" + toString(fp1) + ") = " + toString(tanX));

    int asinX = asin(sinX);
    int acosX = acos(cosX);
    int atanX = atan(tanX);

    System.out.println("asin(" + toString(sinX) + ") = " + toString(asinX));
    System.out.println("acos(" + toString(cosX) + ") = " + toString(acosX));
    System.out.println("atan(" + toString(tanX) + ") = " + toString(atanX));
  }
 */
/*
  static public void LogTest(int input, double dVal)
  {
    int logVal = log(input);
    int expVal = exp(logVal);
    System.out.println("log(" + toString(input) + ")=" + toString(logVal) +
            " (" + Math.log(dVal) + ") ... exp=" + toString(expVal));
  }

  static public void ExpTest(int input, double dVal)
  {
    int val = exp(input);
    System.out.println("exp(" + toString(input) + ")=" + toString(val) +
            " (" + Math.exp(dVal) + ")");
  }
*/

/*
  static public void main (String[] args)
  {
    //PIOVER4 = fromString("0.7853982");
    //System.out.println("PIOVER4 = " + Integer.toHexString(PIOVER4));
    System.out.println(toString(cosd(fromString("-390.0"))));
    System.out.println(toString(cosd(fromString("-360.0"))));
    System.out.println(toString(cosd(fromString("-330.0"))));
    System.out.println(toString(cosd(fromString("-300.0"))));
    System.out.println(toString(cosd(fromString("-270.0"))));
    System.out.println(toString(cosd(fromString("-240.0"))));
    System.out.println(toString(cosd(fromString("-210.0"))));
    System.out.println(toString(cosd(fromString("-180.0"))));
    System.out.println(toString(cosd(fromString("-150.0"))));
    System.out.println(toString(cosd(fromString("-120.0"))));
    System.out.println(toString(cosd(fromString("-90.0"))));
    System.out.println(toString(cosd(fromString("-60.0"))));
    System.out.println(toString(cosd(fromString("-30.0"))));
    System.out.println(toString(cosd(fromString("0.0"))));
    System.out.println(toString(cosd(fromString("30.0"))));
    System.out.println(toString(cosd(fromString("60.0"))));
    System.out.println(toString(cosd(fromString("90.0"))));
    System.out.println(toString(cosd(fromString("120.0"))));
    System.out.println(toString(cosd(fromString("150.0"))));
    System.out.println(toString(cosd(fromString("180.0"))));
    System.out.println(toString(cosd(fromString("210.0"))));
    System.out.println(toString(cosd(fromString("240.0"))));
    System.out.println(toString(cosd(fromString("270.0"))));
    System.out.println(toString(cosd(fromString("300.0"))));
    System.out.println(toString(cosd(fromString("330.0"))));
    System.out.println(toString(cosd(fromString("360.0"))));
    System.out.println(toString(cosd(fromString("390.0"))));

    System.out.println(toString(atand(fromString("-5000.0"))));
    System.out.println(toString(atand(fromString("-5.0"))));
    System.out.println(toString(atand(fromString("-1.0"))));
    System.out.println(toString(atand(fromString("-0.999"))));
    System.out.println(toString(atand(fromString("0.0"))));
    System.out.println(toString(atand(fromString("0.999"))));
    System.out.println(toString(atand(fromString("1.0"))));
    System.out.println(toString(atand(fromString("5.0"))));
    System.out.println(toString(atand(fromString("5000.0"))));

    LogTest(fromString("2000.0"), 2000.0);
    LogTest(fromString("200.0"), 200.0);
    LogTest(fromString("20.0"), 20.0);
    LogTest(fromString("2.0"), 2.0);
    LogTest(fromString("0.2"), 0.2);
    LogTest(fromString("0.02"), 0.02);
    LogTest(fromString("0.002"), 0.002);
    LogTest(fromString("0.0002"), 0.0002);
    LogTest(fromString("0.00002"), 0.00002);
    LogTest(fromString("0.000002"), 0.000002);
    LogTest(fromString("0.0000002"), 0.0000002);
    LogTest(fromString("0.00000002"), 0.00000002);
    LogTest(fromString("0.000000002"), 0.000000002);
    LogTest(fromString("0.0000000002"), 0.0000000002);
    LogTest(fromString("0.00000000002"), 0.00000000002);
    LogTest(fromString("0.000000000002"), 0.000000000002);
    LogTest(fromString("0.0000000000002"), 0.0000000000002);
    LogTest(fromString("0.00000000000002"), 0.00000000000002);
    //ExpTest(fromString("-10.81977828"), -10.81977828);
  }
*/
}
