package com.mobilebasic;

/*
 * Mobile BASIC 1.9 Copyright (c) 2003-2010, David Firth
 *
 * This software is released as Open Source and you are free to modify it
 * and build derived versions for your own use. If you have made any changes
 * that you would like included in the main version then they should be
 * sent to the address below.
 *
 * Patches: E-mail to david@mobilebasic.com
 * Website: http://www.mobilebasic.com/
 */

import javax.microedition.rms.*;
import javax.microedition.io.*;

import java.io.*;

class RandomAccessFile implements DataInput, DataOutput
{
  private static final int MAGIC = 0x52410001; // File Format Magic (RA 0x0001)
  private static final int RESERVED = 8;
  private static final int RECORD_SIZE = 256;

  private static final boolean DEBUG = false;

  private static Main main;

  RecordStore recordStore;
  byte[] recordData;
  boolean recordModified;
  int currentRecord;
  int magic;
  int fileSize;
  int fileOffset;
  int recordSize;

  private void ReadRecord(int recordNumber)
  {
    if (DEBUG)
      System.out.println("ReadRecord: " + recordNumber);

    try
    {
      recordData = recordStore.getRecord(recordNumber);
    }
    catch (RecordStoreException e)
    {
      recordData = new byte[recordSize];

      if (DEBUG)
        System.out.println(e.getClass().getName());
    }
  }

  private void WriteRecord(int recordNumber)
  {
    if (DEBUG)
      System.out.println("WriteRecord: " + recordNumber);

    try
    {
      recordStore.setRecord(recordNumber, recordData, 0, recordSize);
    }
    catch (InvalidRecordIDException e)
    {
      try
      {
        int recnum;

        do
        {
          recnum = recordStore.addRecord(recordData, 0, recordSize);
        } while (recnum != recordNumber);
      }
      catch (RecordStoreFullException e2)
      {
        if (DEBUG)
          System.out.println(e2.getClass().getName());
        throw new BasicError(BasicError.IO_ERROR, "Disk full");
      }
      catch (Exception e2)
      {
        if (DEBUG)
          System.out.println(e2.getClass().getName());
        throw new BasicError(BasicError.IO_ERROR, e2.getClass().getName());
      }
    }
    catch (RecordStoreFullException e)
    {
      throw new BasicError(BasicError.IO_ERROR, "Disk full");
    }
    catch (Exception e)
    {
      if (DEBUG)
        System.out.println(e.getClass().getName());
      throw new BasicError(BasicError.IO_ERROR, e.getClass().getName());
    }
  }

  RandomAccessFile(String filename, boolean readOnly, Main tMain)
  {
    main = tMain;

    if (DEBUG)
      System.out.println("RandomAccessFile() called");

    recordSize = RECORD_SIZE;

    try
    {
      recordStore = RecordStore.openRecordStore(filename, false);
      recordData = null;
      currentRecord = 0;
      recordModified = false;
      fileOffset = 0;
      fileSize = RESERVED; // If fileSize is 0 then readByte() will throw an EOF
      magic = readInt();
      if (magic == MAGIC)
      {
        if (DEBUG)
          System.out.println("MAGIC Matched - File valid");
        fileSize = readInt();
        fileOffset = RESERVED;
      }
      else
      {
        if (DEBUG)
          System.out.println("MAGIC not Matched - What now?");

        fileSize = recordData.length;
        recordSize = fileSize;
        fileOffset = 0;
      }
    }
    catch (RecordStoreNotFoundException e)
    {
      if (!readOnly)
      {
        try
        {
          recordStore = RecordStore.openRecordStore(filename, true);
          recordData = null;
          currentRecord = 0;
          recordModified = false;
          fileSize = RESERVED;
          fileOffset = RESERVED;
          magic = MAGIC;
        }
        catch (Exception e2)
        {
          if (DEBUG)
            System.out.println(e.getClass().getName());
          throw new BasicError(BasicError.IO_ERROR, e.getClass().getName());
        }
      }
      else
      {
        throw new BasicError(BasicError.FILE_NOT_FOUND, "\"" + filename + "\" not found");
      }
    }
    catch (Exception e)
    {
      if (DEBUG)
        System.out.println(e.getClass().getName());
      throw new BasicError(BasicError.IO_ERROR, e.getClass().getName());
    }

    if (DEBUG)
    {
      System.out.println("filename = " + filename);
      System.out.println("fileOffset = " + fileOffset);
      System.out.println("recordSize = " + recordSize);
    }
  }

  void close() throws IOException
  {
    try
    {
      if (magic == MAGIC)
      {
        fileOffset = 0;
        writeInt(MAGIC);
        if (DEBUG)
          System.out.println("fileSize = " + fileSize);
        writeInt(fileSize);
        WriteRecord(currentRecord);
      }
    }
    finally
    {
      try
      {
        recordStore.closeRecordStore();
      }
      catch (RecordStoreException e)
      {
      }
    }
  }

  public byte readByte() throws IOException
  {
    byte byteValue;

    if (fileOffset >= 0)
    {
      if (fileOffset < fileSize)
      {
        int recordNumber = fileOffset / recordSize + 1;
        int recordOffset = fileOffset % recordSize;

        if (recordNumber != currentRecord)
        {
          if (recordModified)
            WriteRecord(currentRecord);

          ReadRecord(recordNumber);
          currentRecord = recordNumber;
          recordModified = false;
        }

        byteValue = recordData[recordOffset];
        fileOffset++;
      }
      else
      {
        throw new EOFException();
      }
    }
    else
    {
      throw new BasicError(BasicError.IO_ERROR, "Invalid offset");
    }

    if (DEBUG)
      System.out.println("readByte " + (fileOffset-1) + " " + byteValue);

    return byteValue;
  }

  public void writeByte(int byteValue) throws IOException
  {
    if (DEBUG)
      System.out.println("writeByte " + fileOffset + " " + byteValue);

    if (fileOffset >= 0)
    {
      int recordNumber = fileOffset / recordSize + 1;
      int recordOffset = fileOffset % recordSize;

      if (recordNumber != currentRecord)
      {
        if (recordModified)
          WriteRecord(currentRecord);

        ReadRecord(recordNumber);
        currentRecord = recordNumber;
        recordModified = false;
      }

      recordData[recordOffset] = (byte)byteValue;
      recordModified = true;
      fileOffset++;

      if (fileOffset >= fileSize)
         fileSize = fileOffset;
    }
    else
    {
      throw new BasicError(BasicError.IO_ERROR, "Invalid offset");
    }
  }

  public boolean readBoolean() throws IOException
  {
    // Not needed by MIDlet
    if (DEBUG)
      System.out.println("readBoolean() not implemented");
    throw new IOException("function not implemented");
  }

  public char readChar() throws IOException
  {
    // Not needed by MIDlet
    if (DEBUG)
      System.out.println("readChar() not implemented");
    throw new IOException("function not implemented");
  }

  public void readFully(byte[] b) throws IOException
  {
    // Used by readUTF()

    for (int i=0;i<b.length;i++)
      b[i] = readByte();
  }

  public void readFully(byte[] b, int off, int len) throws IOException
  {
    // Not needed by MIDlet

    for (int i=off;i<len;i++)
      b[i] = readByte();
  }

  public int readInt() throws IOException
  {
    byte a = readByte();
    byte b = readByte();
    byte c = readByte();
    byte d = readByte();

    int intVal = ((a & 0xff) << 24) |
                 ((b & 0xff) << 16) |
                 ((c & 0xff) << 8) |
                 (d & 0xff);

    return intVal;
  }

  public long readLong() throws IOException
  {
    // Not needed by MIDlet
    if (DEBUG)
      System.out.println("readLong() not implemented");
    throw new IOException("function not implemented");
  }

  public short readShort() throws IOException
  {
    byte a = readByte();
    byte b = readByte();

    int intVal = ((a & 0xff) << 8) |
                 (b & 0xff);

    return (short)intVal;
  }

  public int readUnsignedByte() throws IOException
  {
    // Not needed by MIDlet
    if (DEBUG)
      System.out.println("readUnsignedByte() not implemented");
    throw new IOException("function not implemented");
  }

  public int readUnsignedShort() throws IOException
  {
    // Used by readUTF()

    byte a = readByte();
    byte b = readByte();

    int uShortVal = ((a & 0xff) << 8) |
                    (b & 0xff);

    return uShortVal;
  }

  public String readUTF() throws IOException
  {
    String s = null;
    int tmp = fileOffset;
    int len = readUnsignedShort();

    fileOffset = tmp;
    byte[] byteArray = new byte[len+2];
    readFully(byteArray);

    ByteArrayInputStream inputStream = new ByteArrayInputStream(byteArray);
    DataInputStream in = new DataInputStream(inputStream);
    s = in.readUTF();

    return s;
  }

  public int skipBytes(int n) throws IOException
  {
    // Not needed by MIDlet
    if (DEBUG)
      System.out.println("skipBytes(int n) not implemented");
    throw new IOException("function not implemented");
  }

  public void write(byte[] b) throws IOException
  {
    // Not needed by MIDlet
    //System.out.println("RandomAccessFile.write(byte[]) not implemented");
    if (DEBUG)
      System.out.println("write(byte[]  b) not implemented");
    throw new IOException("function not implemented");
  }

  public void write(byte[] b, int off, int len) throws IOException
  {
    for (int i=off;i<len;i++)
      writeByte(b[i]);
  }

  public void write(int b) throws IOException
  {
    // Not needed by MIDlet
    //System.out.println("RandomAccessFile.write(int b) not implemented");
    if (DEBUG)
      System.out.println("write(int b) not implemented");
    throw new IOException("function not implemented");
  }

  public void writeBoolean(boolean v) throws IOException
  {
    // Not needed by MIDlet
    if (DEBUG)
      System.out.println("writeBoolean not implemented");
    throw new IOException("function not implemented");
  }

  public void writeChar(int v) throws IOException
  {
    // Not needed by MIDlet
    if (DEBUG)
      System.out.println("writeChar not implemented");
    throw new IOException("function not implemented");
  }

  public void writeChars(String s) throws IOException
  {
    // Not needed by MIDlet
    if (DEBUG)
      System.out.println("writeChars not implemented");
    throw new IOException("function not implemented");
  }

  public void writeInt(int v) throws IOException
  {
    writeByte((v >> 24) & 0xff);
    writeByte((v >> 16) & 0xff);
    writeByte((v >> 8) & 0xff);
    writeByte(v & 0xff);
  }

  public void writeLong(long v) throws IOException
  {
    // Not needed by MIDlet
    if (DEBUG)
      System.out.println("writeLong not implemented");
    throw new IOException("function not implemented");
  }

  public void writeShort(int v) throws IOException
  {
    writeByte((v >> 8) & 0xff);
    writeByte(v & 0xff);
  }

  public void writeUTF(String s) throws IOException
  {
    ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
    DataOutputStream out = new DataOutputStream(outputStream);
    out.writeUTF(s);

    byte[] byteArray = outputStream.toByteArray();
    for (int i=0;i<byteArray.length;i++)
      writeByte(byteArray[i]);
  }

  public int getFilePointer()
  {
    return fileOffset - RESERVED;
  }

  public void seek(int offset)
  {
    if (offset >= 0)
    {
      fileOffset = offset + RESERVED;
      if (fileOffset > fileSize)
        fileOffset = fileSize;
    }
    else
    {
      fileOffset = fileSize;
    }
  }
}
